#!/bin/bash
#This script is a template for your analysis

# The <before-rev> is the last compilable revision *before* the error was introduced OR the bug was fixed
# The <after-rev> is the first compilable revision *after* the error was introduced OR the bug was fixed.
# The revision in the <repository directory> is supposed to be the <after-rev>.

if [ "$#" -ne 5 ]; then
  echo "Usage: $0 <repository directory> <subject> <before-revision> <after-revision> <scriptdir>."
  echo The <before-rev> is the last compilable revision *before* the error was introduced OR the bug was fixed
  echo The <after-rev> is the first compilable revision *after* the error was introduced OR the bug was fixed.
  echo The revision in the <repository directory> is supposed to be the <after-rev>.

  exit 1
fi

repo=$1
subject=$2
before_rev=$3
after_rev=$4
scriptdir=$5
results="$scriptdir/results"

function quit {
  echo
  echo $subject_$after_rev [analysis.sh]: $1
  exit 1
}

if ! [ -e "$scriptdir/difftool.sh" ]; then
  quit "difftool.sh missing. Copy the complete CYCC-files into $scriptdir."
fi
if ! [ -e "$scriptdir/csgconstruct" ]; then
  quit "csgconstruct executable missing. Copy the complete CYCC-files into $scriptdir and then run 'make'."
fi



function shorten {
  echo $1 | cut -c -8
}

if ! [ -e "$repo" ] ; then 
 quit "$repo does not exist!"
fi
cd "$repo"
current_rev=$(shorten $(git rev-parse HEAD || quit "Cannot retrieve hash for current revision from $repo!"))
if ! [[ "$current_rev" == *"$after_rev"* ]]; then
  quit "The revision in $repo is supposed to be $after_rev but turns out to be $current_rev.".
fi

if ! [ -e "$results" ]; then
  mkdir "$results" || quit "Cannot create folder $results" 
fi


##############
## ANALYSIS ##
##############

cd $repo
s_before_rev=$(shorten $before_rev)
if [[ "$s_before_rev" == *"$after_rev"* ]]; then
  #only refer to it by $after_rev
  filename="$subject"_$(git rev-list HEAD..master | wc -l)_$after_rev || quit "Problem computing filename."
else
  #refer to it by $before_rev and $after_rev
  filename="$subject"_$(git rev-list HEAD..master | wc -l)_$(shorten $s_before_rev)_$after_rev || quit "Problem computing filename."
fi

git difftool -y -x "$scriptdir/difftool.sh" $before_rev $after_rev > "$results/$filename.diff" || quit "Problem computing difference."
"$scriptdir/csgconstruct" --git $repo --diff notrelevant --use-diff-file "$results/$filename.diff" > "$results/$filename.complexity.txt" 2>&1 || quit "Problem with csgconstruct."
mv callgraph.dot "$results/$filename.cg.dot"
mv intercsg.dot "$results/$filename.csg.dot"

cycc=$(grep "Interprocedural Change Compl." "$results/$filename.complexity.txt" | cut -c 33- )
cloc=$(grep "Number of changed linesOfCode" "$results/$filename.complexity.txt" | cut -c 33- | cut -d " " -f1 )
if [ $cycc -lt 1 ]; then
  cycc=0
  cloc=0
fi
echo "${subject}_${before_rev}_$after_rev: CyCC=$cycc CLoC=$cloc"


