import java.io.*;
import java.util.*;

/**
 * This class implements the cubic time and quadratic time algorithms for 
 * computing the predictions maximizing the expected F-score under the label
 * independence assumption. The algorithms are described in 
 * <i>Nan Ye, Adam K.M. Chai, Wee Sun Lee, Hai Leong Chieu. Optimizing 
 * F-measures: A Tale of Two Approaches. ICML 2012.</i>
 *
 * In this implementation, <i>F<sub>beta</sub> = (1+beta)/(1/prec+beta/rec)
 * </i>, where prec and rec are the precision and recall respectively.
 *
 * @author <a href="mailto:yenan@comp.nus.edu.sg">Ye Nan</a>
 */
public class FScore {
	private static class PairComp implements Comparator<double[]> {
		public PairComp() {}
		public int compare(double[] ar1, double[] ar2) {
			if(ar1[0] > ar2[0]) return -1;
			if(ar1[0] < ar2[0]) return 1;
			return 0;
		}

		public boolean equals(double[] ar1, double[] ar2) { return ar1[0] == ar2[0]; }
	}

	/**
	 * Sort p in descending order, and return the corresponding ordering of
	 * elements in p.
	 */
	private static int[] sort(double[] p) {
		int N = p.length;
		double[][] indexed = new double[N][2];
		for(int i=0; i<N; i++) {
			indexed[i][0] = p[i];
			indexed[i][1] = i;
		}
		Arrays.sort(indexed, new PairComp());
		int[] perm = new int[N];
		for(int i=0; i<N; i++) {
			p[i] = indexed[i][0];
			perm[i] = (int) indexed[i][1];
		}
		return perm;
	}
	private static double max(double[] ar) {
		double max = Double.NEGATIVE_INFINITY;
		for(double d : ar) 
			if(d > max)
				max = d;
		return max;
	}
	private static int max_idx(double[] ar) {
		double max = Double.NEGATIVE_INFINITY;
		int idx = -1;
		for(int i=0; i<ar.length; i++)
			if(ar[i] > max) {
				max = ar[i];
				idx = i;
			}
		return idx;
	}
	private static void perm(double[] ar, int[] perm) {
		int N = ar.length; 
		double[] copy = new double[N];
		for(int i=0; i<N; i++) copy[i] = ar[perm[i]];
		for(int i=0; i<N; i++) ar[i] = copy[i];
	}

	/**
	 * Return the maximum expected F<sub>q/r</sub> when the probabilities that the 
	 * instances are positive are given by p
	 */
	public static double max_expected_fscore(double[] p, int q, int r) { 
		int[] perm = sort(p);
		double f = max(expected_fscores(p, q, r)); 
		perm(p, perm);
		return f;
	}
	/**
	 * Return the predictions maximizing the expected F<sub>q/r</sub> when the 
	 * probabilities that the instances are positive are given by p.
	 */
	public static int[] max_expected_fscore_preds(double[] p, int q, int r) {
		int N = p.length;
		int[] perm = sort(p);
		int R = max_idx(expected_fscores(p, q, r));
		int[] preds = new int[N];
		for(int i=0; i<R; i++) preds[perm[i]] = 1;
		perm(p, perm);
		return preds;
	}
	/**
	 * Return the F<sub>q/r</sub> scores when the first k instances are predicted as
	 * positive, for all k.
	 */
	public static double[] expected_fscores(double[] p, int q, int r) {
		int nInsts = p.length;
		double[] fs = new double[nInsts+1];
		double beta = 1.0*q/r;
	
		int A = 0, B = q+r;
		double[] sums = new double[(q+r)*nInsts+2];
		for(int i=1; i<=B*nInsts; i++) sums[i] = 1.0*q/i;

		double[][] poly = new double[nInsts+1][]; //[p_1x+(1-p_1)][p_2x+(1-p_2)]...[p_nx+(1-p_n)]
		poly[0] = new double[]{0,1,0};
		for(int i=0; i<nInsts; i++) {
			poly[i+1] = new double[i+4];
			for(int j=i+2; j>0; j--)
				poly[i+1][j] = (1-p[i])*poly[i][j]+p[i]*poly[i][j-1];
		}

		for(int n=nInsts; n>0; n--) {
			for(int TP=0; TP<=n; TP++) fs[n] += ((1+beta)/beta) * TP * poly[n][TP+1] * sums[n*r+TP*q];

			for(int i=1; i<=B*(n-1); i++) sums[i] = (1-p[n-1])*sums[i]+p[n-1]*sums[i+q];
		}

		fs[0] = expected_fscore(p, 0, beta); 

		return fs;
	}
	
	/**
	 * Return the maximum expected F<sub>beta</sub> when the probabilities that the 
	 * instances are positive are given by p
	 */
	public static double max_expected_fscore(double[] p, double beta) { 
		int[] perm = sort(p);
		double f = max(expected_fscores(p, beta)); 
		perm(p, perm);
		return f;
	}
	/**
	 * Return the predictions maximizing the expected F<sub>beta</sub> when the 
	 * probabilities that the instances are positive are given by p.
	 */
	public static int[] max_expected_fscore_preds(double[] p, double beta) {
		int N = p.length;
		int[] perm = sort(p);
		int R = max_idx(expected_fscores(p, beta));
		int[] preds = new int[N];
		for(int i=0; i<R; i++) preds[perm[i]] = 1;
		perm(p, perm);
		return preds;
	}
	/**
	 * Return the F<sub>beta</sub> scores when the first n instances are predicted as
	 * positive, for all n.
	 */
	public static double[] expected_fscores(double[] p, double beta) {
		double[] fs = new double[p.length+1];
		for(int n=0; n<=p.length; n++) fs[n] = expected_fscore(p, n, beta);
		return fs;
	}
	/**
	 * Return the F<sub>beta</sub> score when the first n instances are predicted as
	 * positive.
	 */
	public static double expected_fscore(double[] p, int n, double beta) {
		int nInsts = p.length;
		
		if(n==0) {
			double degenerateF1 = 1;
			for(int i=0; i<nInsts; i++) degenerateF1 *= (1-p[i]);
			return degenerateF1;
		}
		
		double[] poly1 = new double[nInsts+2]; //[p_1x+(1-p_1)][p_2x+(1-p_2)]...[p_nx+(1-p_n)]
		poly1[1] = 1; 
		for(int i=0; i<n; i++)
			for(int j=i+2; j>0; j--)
				poly1[j] = (1-p[i])*poly1[j]+p[i]*poly1[j-1];
	
		double[] poly2 = new double[nInsts+2]; //[p_{n+1}x+(1-p_{n+1})]...[p_Nx+(1-p_N)]
		poly2[1] = 1; 
		for(int i=0; i<nInsts-n; i++)
			for(int j=i+2; j>0; j--)
				poly2[j] = (1-p[n+i])*poly2[j]+p[n+i]*poly2[j-1];

		double E = 0;
		for(int TP=0; TP<=n; TP++) 
			for(int FN=0; FN<=nInsts-n; FN++)
				E += Math.abs(poly1[1+TP]*poly2[1+FN])*(1+beta)*TP/(n+beta*(TP+FN));

		return E;
	}

	public static void test(double[] p, int q, int r) {
		int N = p.length;
		System.out.println("q="+q+" r="+r);
		System.out.print("p: ");
		for(int i=0; i<N; i++)
			System.out.print(" "+p[i]);
		System.out.println();

		double[] f1 = expected_fscores(p, q, r);
		double[] f2 = expected_fscores(p, 1.0*q/r);
		System.out.println("F-scores for choosing top instances as positive:");
		System.out.print("  Quad:");
		for(int i=0; i<N; i++) 
			System.out.print(" "+f1[i]);
		System.out.println();
		System.out.print("  Cube:");
		for(int i=0; i<N; i++) 
			System.out.print(" "+f2[i]);
		System.out.println();

		System.out.println("Optimal labeling:");
		int[] preds1 = max_expected_fscore_preds(p, q, r);
		System.out.print("  Quad:");
		for(int i=0; i<N; i++) 
			System.out.print(" "+preds1[i]);
		System.out.println();
		int[] preds2 = max_expected_fscore_preds(p, 1.0*q/r);
		System.out.print("  Cube:");
		for(int i=0; i<N; i++) 
			System.out.print(" "+preds2[i]);
		System.out.println("\n");
	}
	public static void main(String [] args) throws IOException {
		double[] p = new double[]{0.4, 0.1, 0.3, 0.4};

		test(p, 1, 1);
		test(p, 1, 3);
		test(p, 3, 1);
	}
}
