#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "Global.h"
#include "cfptree_outbuf.h"


//------------------------------------------------------
//Initialize variables used for cfp-tree read and write
//--------------------------------------------------------
void CTreeOutBufManager::Init()
{
	char sztree_filename[200];

	//=================================================
	//set buffer page size and buffer size
	mntree_buf_page_size = DEFAULT_TREE_BUF_PAGE_SIZE;
	while((unsigned int)mntree_buf_page_size<sizeof(ENTRY)+sizeof(int)*(gnum_of_attrs+1)+gntgt_stat_size)
		mntree_buf_page_size *= 2;
	while((unsigned int)mntree_buf_page_size<(sizeof(ENTRY)+gntgt_stat_size)*gnum_of_items+2*sizeof(int))
			mntree_buf_page_size *= 2;
	mntree_outbuf_size = DEFAULT_TREE_OUTBUF_SIZE;
	mntree_buf_num_of_pages = (mntree_outbuf_size-1)/mntree_buf_page_size+1;
	if(mntree_buf_num_of_pages==1)
		mntree_buf_num_of_pages++;
	mntree_outbuf_size = mntree_buf_page_size*mntree_buf_num_of_pages;
	//=================================================

	mpactive_stack = NewActiveStack(gnum_of_attrs);
	mnactive_top = 0;
	mnmax_tree_depth = 0;
	mndfs_entries_len = 0;
	mnmax_dfsentries_len = 0;
	mndfs_sup_sum = 0;
	mnmax_dfs_sup_sum = 0;

	sprintf(sztree_filename, "%s.cfptree", gszoutput_filename);
	mfpcfp_file = fopen(sztree_filename, "wb");
	if(mfpcfp_file==NULL)
	{
		printf("Error: cannot open file %s for write\n", sztree_filename);
		exit(-1);
	}

	mptree_buffer = NewCFPBufPointers(mntree_buf_num_of_pages);
	memset(mptree_buffer, 0, sizeof(char*)*mntree_buf_num_of_pages);
	mncapacity = mntree_outbuf_size-sizeof(int);
	mnpage_size = mntree_buf_page_size-sizeof(int);

	mndisk_start_pos = 0;
	mndisk_write_start_pos = -1;
	mncur_page_no = 0;
	mncur_page_pos = 0;
	mntree_size = 0;
	mnmin_inmem_level = -1;
	mnpreorder = 0;

	mfpread_file = fopen(sztree_filename, "rb");
	if(mfpread_file==NULL)
	{
		printf("Error: cannot open file %s for read\n", sztree_filename);
		exit(-1);
	}

	mpitem_bitmap = NewCharArray(gnum_of_items);
	memset(mpitem_bitmap, 0, sizeof(char)*gnum_of_items);
}

//------------------------------------------------------------------
//Release space occupied by variables after finish mining
//------------------------------------------------------------------
void CTreeOutBufManager::Destroy()
{
	DumpAll();

	DelActiveStack(mpactive_stack, gnum_of_attrs);
	for(int i=0;i<mntree_buf_num_of_pages;i++)
	{
		if(mptree_buffer[i]!=NULL)
			DelCFPBufPage(mptree_buffer[i], mntree_buf_page_size);
	}
	DelCFPBufPointers(mptree_buffer, mntree_buf_num_of_pages);

	fclose(mfpcfp_file);
	fclose(mfpread_file);
	OutputTreeStatis();

	DelCharArray(mpitem_bitmap, gnum_of_items);

}

void CTreeOutBufManager::PrintfStatistics(FILE *fpsum_file)
{
	fprintf(fpsum_file, "%.2fMB\t", (double)mntree_size/(1<<20));
}

void CTreeOutBufManager::OutputTreeStatis()
{
	char szstat_filename[200];
	FILE *fp_stat;
	int i;

	sprintf(szstat_filename, "%s.cfptree.stat", gszoutput_filename);	
	fp_stat = fopen(szstat_filename, "wt");
	if(fp_stat==NULL)
	{
		printf("Error: cannot open file %s for write\n", szstat_filename);
		return;
	}
	fprintf(fp_stat, "%d %d %d\n", gndb_size, gnmin_sup, gnmax_sup);
	fprintf(fp_stat, "%d %d %d\n", gnum_of_attrs-1, gnum_of_items, gnum_of_freq_items);
	fprintf(fp_stat, "%d %d\n", mntree_size, mntree_buf_page_size);
	fprintf(fp_stat, "%d\n", mnpreorder);
	fprintf(fp_stat, "%d %d\n", mnmax_tree_depth, gnmax_pattern_len);
	fprintf(fp_stat, "%d %d\n", mnmax_dfsentries_len, mnmax_dfs_sup_sum);
	fprintf(fp_stat, "%d\n", gntgt_attr_type);
	if(gntgt_attr_type==CONTINUOUS || gntgt_attr_type==CONTINUOUS_NORMAL)
		fprintf(fp_stat, "%.2f %.2f\n", gtgt_stat.ptgt_sum->dsum, gtgt_stat.ptgt_sum->dsquare_sum);
	else if(gsztarget_value[0]!=0)
		fprintf(fp_stat, "%d\n", gtgt_stat.ntgt_sup);
	else
	{
		fprintf(fp_stat, "%d ", gnum_of_tgt_values);
		for(i=0;i<gnum_of_tgt_values;i++)
			fprintf(fp_stat, "%d ", gtgt_stat.ptgt_sups[i]);
		fprintf(fp_stat, "\n");
	}
	

	fclose(fp_stat);

	printf("CFP-tree size: %d\n", mntree_size);
	printf("#frequent entries: %d\n", mnpreorder);
}

int CTreeOutBufManager::GetTreeSize()
{
	return mntree_size;
}


//------------------------------------------------------------------------
//
//------------------------------------------------------------------------
int CTreeOutBufManager::FillPage(int node_size)
{
	int nfill_size;

	nfill_size = 0;
	if(mncur_page_pos>0 && mncur_page_pos+node_size>mnpage_size)
	{
		nfill_size = mnpage_size-mncur_page_pos;
		if(mntree_size+nfill_size-mndisk_start_pos>=mncapacity)
			Dump(nfill_size);

		if(mptree_buffer[mncur_page_no]==NULL)
			mptree_buffer[mncur_page_no] = NewCFPBufPage(mntree_buf_page_size);
		memcpy(&(mptree_buffer[mncur_page_no][mnpage_size]), &nfill_size, sizeof(int));
		if(nfill_size>0)
			memset(&(mptree_buffer[mncur_page_no][mncur_page_pos]), -1, nfill_size);
		nfill_size += sizeof(int);
		mncur_page_no = (mncur_page_no+1)%mntree_buf_num_of_pages;
		mncur_page_pos = 0;
		mntree_size += nfill_size;
	}
	return nfill_size;
}

void CTreeOutBufManager::UpdateAncestors(int bitmap)
{
	int top, ncur_order;
	ENTRY *pentry;

	top = mnactive_top-1;
	pentry = NULL;
	if(top>=0)
	{
		ncur_order = mpactive_stack[top].ncur_order;
		mpactive_stack[top].pentries[ncur_order].child = mntree_size;
	}
	while(top>=0 && mpactive_stack[top].num_of_entries<=1)
	{
		mpactive_stack[top].pentries->hash_bitmap |= bitmap;
		top--;
	}
	if(top>=0)
	{
		ncur_order = mpactive_stack[top].ncur_order;
		mpactive_stack[top].pentries[ncur_order].hash_bitmap |= bitmap;
	}
}

//----------------------------------------------------------------------------------
//Insert an internal node into CFP-tree. The child position of entries in this node
//is unknown when the node is inserted. Therefore when a CFP-node is first inserted,
//it will be inserted into the active stack. After all its children node are created 
//and the information in the CFP-node is complete. It will be popped out from the 
//active stack.
//----------------------------------------------------------------------------------
void CTreeOutBufManager::InsertInternalNode(HEADER_NODE* pheader_table, char* ptgt_stat_array, int num_of_items)
{
	int node_size, nfull_node_size, bitmap, i;

	if(mncur_page_no!=(mntree_size/mntree_buf_page_size)%mntree_buf_num_of_pages)
		printf("Error[InsertNode]: Inconsistent cur_page_no\n");
	if(mncur_page_pos!=mntree_size%mntree_buf_page_size)
		printf("Error[InsertNode]: Inconsistent cur_page_pos\n");

	node_size = sizeof(int)+sizeof(ENTRY)*num_of_items;
	nfull_node_size = node_size+gntgt_stat_size*num_of_items;
	FillPage(nfull_node_size);

	if(mntree_size+nfull_node_size-mndisk_start_pos>=mncapacity)
		Dump(nfull_node_size);

	bitmap = 0;
	for(i=0;i<num_of_items;i++)
		bitmap = bitmap | (1<<pheader_table[i].nitem%CFP_HASH_LEN);
	UpdateAncestors(bitmap);

	mpactive_stack[mnactive_top].ndisk_pos = mntree_size;
	mpactive_stack[mnactive_top].in_mem = true;
	mpactive_stack[mnactive_top].mem_page_no = mncur_page_no;
	mpactive_stack[mnactive_top].mem_page_pos = mncur_page_pos;
	if(mnmin_inmem_level==-1)
		mnmin_inmem_level = mnactive_top;

	mpactive_stack[mnactive_top].cfpnode_size = node_size;
	mpactive_stack[mnactive_top].nfull_node_size = nfull_node_size;

	mpactive_stack[mnactive_top].num_of_entries = num_of_items;
	mpactive_stack[mnactive_top].ncur_order = 0;
	if(mptree_buffer[mncur_page_no]==NULL)
		mptree_buffer[mncur_page_no] = NewCFPBufPage(mntree_buf_page_size);
	memcpy(&(mptree_buffer[mncur_page_no][mncur_page_pos]), &num_of_items, sizeof(int));
	mncur_page_pos += sizeof(int);
	mpactive_stack[mnactive_top].pentries = (ENTRY*)&(mptree_buffer[mncur_page_no][mncur_page_pos]);
	mpactive_stack[mnactive_top].nsup_sum = 0;
	for(i=0;i<num_of_items;i++)
	{
		mpactive_stack[mnactive_top].pentries[i].item = pheader_table[i].nitem;
		mpactive_stack[mnactive_top].pentries[i].support = pheader_table[i].nsupport;
		mpactive_stack[mnactive_top].pentries[i].child = 0;
		mpactive_stack[mnactive_top].pentries[i].hash_bitmap = 0;
		mpactive_stack[mnactive_top].pentries[i].npreorder = -1;
		mpactive_stack[mnactive_top].nsup_sum += pheader_table[i].nsupport;
	}
	mncur_page_pos += sizeof(ENTRY)*num_of_items;

	memcpy(&mptree_buffer[mncur_page_no][mncur_page_pos], ptgt_stat_array, gntgt_stat_size*num_of_items);
	mncur_page_pos += gntgt_stat_size*num_of_items;

	mntree_size += nfull_node_size;

	if(mpactive_stack[mnactive_top].pentries[0].support>=gnmin_sup)
	{
		mpactive_stack[mnactive_top].pentries[0].npreorder = mnpreorder;
		mnpreorder++;
	}

	if(mnactive_top==0)
		mpactive_stack[mnactive_top].nprefix_len = 1;
	else
		mpactive_stack[mnactive_top].nprefix_len = mpactive_stack[mnactive_top-1].nprefix_len+1;

	mndfs_entries_len += num_of_items;
	if(mnmax_dfsentries_len<mndfs_entries_len)
		mnmax_dfsentries_len = mndfs_entries_len;
	mndfs_sup_sum += mpactive_stack[mnactive_top].nsup_sum;
	if(mnmax_dfs_sup_sum<mndfs_sup_sum)
		mnmax_dfs_sup_sum = mndfs_sup_sum;

	mnactive_top++;
	if(mnmax_tree_depth<mnactive_top)
		mnmax_tree_depth = mnactive_top;
}

void CTreeOutBufManager::MoveNext()
{
	int ncur_pos;

	mpactive_stack[mnactive_top-1].ncur_order++;
	ncur_pos = mpactive_stack[mnactive_top-1].ncur_order;
	if(ncur_pos<mpactive_stack[mnactive_top-1].num_of_entries && mpactive_stack[mnactive_top-1].pentries[ncur_pos].support>=gnmin_sup)
	{
		mpactive_stack[mnactive_top-1].pentries[ncur_pos].npreorder = mnpreorder;
		mnpreorder++;
	}
	
}

void CTreeOutBufManager::WriteInternalNode()
{
	mnactive_top--;
	mndfs_entries_len -= mpactive_stack[mnactive_top].num_of_entries;
	mndfs_sup_sum -= mpactive_stack[mnactive_top].nsup_sum;

	if(mnmin_inmem_level>=mnactive_top)
		mnmin_inmem_level = -1;

	if(!mpactive_stack[mnactive_top].in_mem)
	{
		int nfile_pos;

		nfile_pos = ftell(mfpcfp_file);
		if(nfile_pos!=mpactive_stack[mnactive_top].ndisk_pos)
			fseek(mfpcfp_file, mpactive_stack[mnactive_top].ndisk_pos-nfile_pos, SEEK_CUR);

		fwrite(&(mpactive_stack[mnactive_top].num_of_entries), sizeof(int), 1, mfpcfp_file);
		fwrite(mpactive_stack[mnactive_top].pentries, sizeof(ENTRY), mpactive_stack[mnactive_top].num_of_entries, mfpcfp_file);
		DelEntryArray(mpactive_stack[mnactive_top].pentries, mpactive_stack[mnactive_top].num_of_entries);

		fflush(mfpcfp_file);
		
		return;
	}
	else
	{
		if(mndisk_write_start_pos==-1 || mndisk_write_start_pos>mpactive_stack[mnactive_top].ndisk_pos)
			mndisk_write_start_pos = mpactive_stack[mnactive_top].ndisk_pos;
	}
}

//-----------------------------------------------------------------------------------------
//Insert a CFP-tree node into output buffer and active stack. This node either appears in 
//every transaction of a conditional database, or is the only child of a CFP-tree root.
//-----------------------------------------------------------------------------------------
void CTreeOutBufManager::InsertCommonPrefixNode(int *pitems, int num_of_items, int nsup, char* ptgt_stat)
{
	int node_size, nfull_node_size, bitmap, i;

	if(mncur_page_no!=(mntree_size/mntree_buf_page_size)%mntree_buf_num_of_pages)
		printf("Error[InsertNode]: Inconsistent cur_page_no\n");
	if(mncur_page_pos!=mntree_size%mntree_buf_page_size)
		printf("Error[InsertNode]: Inconsistent cur_page_pos\n");

	if(num_of_items==1)
		node_size = sizeof(int) + sizeof(ENTRY);
	else 
		node_size = num_of_items*sizeof(int)+sizeof(ENTRY);
	nfull_node_size = node_size+gntgt_stat_size;

	FillPage(nfull_node_size);

	if(mntree_size+nfull_node_size-mndisk_start_pos>=mncapacity)
		Dump(nfull_node_size);

	bitmap = 0;
	for(i=0;i<num_of_items;i++)
		bitmap = bitmap | (1<<pitems[i]%CFP_HASH_LEN);
	UpdateAncestors(bitmap);

	mpactive_stack[mnactive_top].ndisk_pos = mntree_size;
	mpactive_stack[mnactive_top].in_mem = true;
	mpactive_stack[mnactive_top].mem_page_no = mncur_page_no;
	mpactive_stack[mnactive_top].mem_page_pos = mncur_page_pos;
	if(mnmin_inmem_level==-1)
		mnmin_inmem_level = mnactive_top;

	if(mptree_buffer[mncur_page_no]==NULL)
		mptree_buffer[mncur_page_no] = NewCFPBufPage(mntree_buf_page_size);
	mpactive_stack[mnactive_top].cfpnode_size = node_size;
	mpactive_stack[mnactive_top].nfull_node_size = nfull_node_size;
	if(num_of_items>1)
		mpactive_stack[mnactive_top].num_of_entries = -num_of_items;
	else
		mpactive_stack[mnactive_top].num_of_entries = 1;
	mpactive_stack[mnactive_top].ncur_order = 0;
	memcpy(&(mptree_buffer[mncur_page_no][mncur_page_pos]), &(mpactive_stack[mnactive_top].num_of_entries), sizeof(int));
	mncur_page_pos += sizeof(int);

	mpactive_stack[mnactive_top].pentries = (ENTRY*)&(mptree_buffer[mncur_page_no][mncur_page_pos]);
	mpactive_stack[mnactive_top].pentries[0].support = nsup;
	mpactive_stack[mnactive_top].pentries[0].child = 0;
	mpactive_stack[mnactive_top].pentries[0].hash_bitmap = 0;
	mpactive_stack[mnactive_top].nsup_sum = nsup;
	if(nsup>=gnmin_sup)
	{
		mpactive_stack[mnactive_top].pentries[0].npreorder = mnpreorder;
		mnpreorder++;
	}
	mncur_page_pos += sizeof(ENTRY)-sizeof(int);

	memcpy(&mptree_buffer[mncur_page_no][mncur_page_pos], pitems, sizeof(int)*num_of_items);
	mncur_page_pos += sizeof(int)*num_of_items;

	memcpy(&(mptree_buffer[mncur_page_no][mncur_page_pos]), ptgt_stat, gntgt_stat_size);
	mncur_page_pos += gntgt_stat_size;

	mntree_size += nfull_node_size;

	if(mnactive_top==0)
		mpactive_stack[mnactive_top].nprefix_len = num_of_items;
	else
		mpactive_stack[mnactive_top].nprefix_len = mpactive_stack[mnactive_top-1].nprefix_len+num_of_items;

	mndfs_entries_len++;
	if(mnmax_dfsentries_len<mndfs_entries_len)
		mnmax_dfsentries_len = mndfs_entries_len;

	if(nsup<gndb_size)
	{
		mndfs_sup_sum += nsup;
		if(mnmax_dfs_sup_sum<mndfs_sup_sum)
			mnmax_dfs_sup_sum = mndfs_sup_sum;
	}

	mnactive_top++;
	if(mnmax_tree_depth<mnactive_top)
		mnmax_tree_depth = mnactive_top;
}

void CTreeOutBufManager::WriteCommonPrefixNode(int num_of_nodes)
{
	int i, nfile_pos;
	
	for(i=0;i<num_of_nodes;i++)
	{
		mnactive_top--;
		mndfs_entries_len--;
		if(mpactive_stack[mnactive_top].nsup_sum<gndb_size)
			mndfs_sup_sum -= mpactive_stack[mnactive_top].nsup_sum;

		if(mnmin_inmem_level>=mnactive_top)
			mnmin_inmem_level = -1;

		if(!mpactive_stack[mnactive_top].in_mem)
		{
			nfile_pos = ftell(mfpcfp_file);
			if(mpactive_stack[mnactive_top].ndisk_pos!=nfile_pos)
				fseek(mfpcfp_file, mpactive_stack[mnactive_top].ndisk_pos-nfile_pos, SEEK_CUR);

			fwrite(&(mpactive_stack[mnactive_top].num_of_entries), sizeof(int), 1, mfpcfp_file);
			if(mpactive_stack[mnactive_top].num_of_entries==1)
				fwrite(mpactive_stack[mnactive_top].pentries, sizeof(ENTRY), 1, mfpcfp_file);
			else 
			{
				fwrite(mpactive_stack[mnactive_top].pentries, sizeof(ENTRY)-sizeof(int), 1, mfpcfp_file);
				fwrite(mpactive_stack[mnactive_top].pentries->pitems, sizeof(int), -mpactive_stack[mnactive_top].num_of_entries, mfpcfp_file);
			}
			if(mpactive_stack[mnactive_top].num_of_entries<-1)
				DelCFPItemset(mpactive_stack[mnactive_top].pentries->pitems, -mpactive_stack[mnactive_top].num_of_entries);
			DelEntryArray(mpactive_stack[mnactive_top].pentries, 1);
			fflush(mfpcfp_file);
		}
		else
		{
			if(mndisk_write_start_pos==-1 || mndisk_write_start_pos>mpactive_stack[mnactive_top].ndisk_pos)
				mndisk_write_start_pos = mpactive_stack[mnactive_top].ndisk_pos;
		}
	}
}

//--------------------------------------------------------------------------------------
// Write leaf node
//--------------------------------------------------------------------------------------
void CTreeOutBufManager::WriteLeafNode(int *pitems, int num_of_items, int nsup, char* ptgt_stat)
{
	ENTRY *pentry;
	int node_size, bitmap, i;

	if(num_of_items==1)
		node_size = sizeof(ENTRY)+sizeof(int);
	else
		node_size = sizeof(ENTRY)+sizeof(int)*num_of_items;
	node_size += gntgt_stat_size;
	FillPage(node_size);

	if(mntree_size+node_size-mndisk_start_pos>=mncapacity)
		Dump(node_size);

	bitmap = 0;
	for(i=0;i<num_of_items;i++)
		bitmap = bitmap | (1<<pitems[i]%CFP_HASH_LEN);
	UpdateAncestors(bitmap);

	if(mndisk_write_start_pos==-1 || mndisk_write_start_pos>mntree_size)
		mndisk_write_start_pos = mntree_size;

	if(num_of_items>1)
		num_of_items = -num_of_items;
	if(mptree_buffer[mncur_page_no]==NULL)
		mptree_buffer[mncur_page_no] = NewCFPBufPage(mntree_buf_page_size);
	memcpy(&(mptree_buffer[mncur_page_no][mncur_page_pos]), &num_of_items, sizeof(int));
	mncur_page_pos += sizeof(int);

	pentry = (ENTRY*)&(mptree_buffer[mncur_page_no][mncur_page_pos]);
	pentry->support = nsup;
	pentry->child = 0;
	pentry->hash_bitmap = 0;
	if(nsup>=gnmin_sup)
	{
		pentry->npreorder = mnpreorder;
		mnpreorder++;
	}
	mncur_page_pos += sizeof(ENTRY)-sizeof(int);

	if(num_of_items==1)
	{
		memcpy(&mptree_buffer[mncur_page_no][mncur_page_pos], pitems, sizeof(int));
		mncur_page_pos += sizeof(int);
	}
	else
	{
		memcpy(&mptree_buffer[mncur_page_no][mncur_page_pos], pitems, sizeof(int)*(-num_of_items));
		mncur_page_pos += sizeof(int)*(-num_of_items);
	}

	memcpy(&mptree_buffer[mncur_page_no][mncur_page_pos], ptgt_stat, gntgt_stat_size);
	mncur_page_pos += gntgt_stat_size;

	mntree_size += node_size;

	if(mnmax_tree_depth<mnactive_top+1)
		mnmax_tree_depth = mnactive_top+1;

	if(mnmax_dfsentries_len<mndfs_entries_len+1)
		mnmax_dfsentries_len = mndfs_entries_len+1;
	if(mnmax_dfs_sup_sum<mndfs_sup_sum+nsup)
		mnmax_dfs_sup_sum = mndfs_sup_sum+nsup;
}

//-------------------------------------------------------------------------------------
//When buffer is full, dump some of its contents on disk to release space for furture
//CFP-tree nodes
//-------------------------------------------------------------------------------------
void CTreeOutBufManager::Dump(int newnode_size)
{
	int nstart_page_no, nstart_page_pos, nfull_node_size, nfill_size;
	int nwrite_page_no, nwrite_page_pos, ndisk_write_end_pos, nfile_pos, top;
	int num_of_entries, ntgt_stat_page_pos, ntgt_stat_disk_pos;

	while(mntree_size+newnode_size-mndisk_start_pos>=mncapacity)
	{
		//The first node in the buffer is an active node in the stack. 
		//Take this node out of the buffer, and write its target statistics to disk
		if(mndisk_start_pos!=mndisk_write_start_pos) 
		{
			if(mndisk_start_pos!=mpactive_stack[mnmin_inmem_level].ndisk_pos)
				printf("Error with the starting position %d %d\n", mndisk_start_pos, mpactive_stack[mnmin_inmem_level].ndisk_pos);

			nstart_page_no = mpactive_stack[mnmin_inmem_level].mem_page_no;
			nstart_page_pos = mpactive_stack[mnmin_inmem_level].mem_page_pos;
			
			nfull_node_size = mpactive_stack[mnmin_inmem_level].nfull_node_size;
			num_of_entries = mpactive_stack[mnmin_inmem_level].num_of_entries;

			if(num_of_entries>=1)
			{
				mpactive_stack[mnmin_inmem_level].pentries = NewEntryArray(num_of_entries);
				memcpy(mpactive_stack[mnmin_inmem_level].pentries, &(mptree_buffer[nstart_page_no][nstart_page_pos+sizeof(int)]), sizeof(ENTRY)*num_of_entries);
			}
			else 
			{
				num_of_entries = -num_of_entries;
				mpactive_stack[mnmin_inmem_level].pentries = NewEntryArray(1);
				mpactive_stack[mnmin_inmem_level].pentries->pitems = NewCFPItemset(num_of_entries);
				memcpy(mpactive_stack[mnmin_inmem_level].pentries, &(mptree_buffer[nstart_page_no][nstart_page_pos+sizeof(int)]), sizeof(ENTRY)-sizeof(int));
				memcpy(mpactive_stack[mnmin_inmem_level].pentries->pitems, &(mptree_buffer[nstart_page_no][nstart_page_pos+sizeof(ENTRY)]), sizeof(int)*num_of_entries);
			}

			//write the target statistics to disk
			ntgt_stat_page_pos = nstart_page_pos+mpactive_stack[mnmin_inmem_level].cfpnode_size;
			ntgt_stat_disk_pos = mpactive_stack[mnmin_inmem_level].ndisk_pos+mpactive_stack[mnmin_inmem_level].cfpnode_size;
			nfile_pos = ftell(mfpcfp_file);
			if(ntgt_stat_disk_pos!=nfile_pos)
				fseek(mfpcfp_file, ntgt_stat_disk_pos-nfile_pos, SEEK_CUR);
			fwrite(&mptree_buffer[nstart_page_no][ntgt_stat_page_pos], sizeof(char), gntgt_stat_size, mfpcfp_file); 


			memcpy(&nfill_size, &(mptree_buffer[nstart_page_no][mnpage_size]), sizeof(int));
			nstart_page_pos += nfull_node_size;
			mndisk_start_pos += nfull_node_size;
			if(nstart_page_pos==mnpage_size)
				mndisk_start_pos += sizeof(int);
			else if(nstart_page_pos+nfill_size==mnpage_size)
				mndisk_start_pos += nfill_size+sizeof(int);

			mpactive_stack[mnmin_inmem_level].in_mem = false;
			mpactive_stack[mnmin_inmem_level].mem_page_no = -1;
			mpactive_stack[mnmin_inmem_level].mem_page_pos = -1;
			mnmin_inmem_level++;
			if(mnmin_inmem_level>=mnactive_top)
				mnmin_inmem_level = -1;
		}
		//the first node in the buffer is not an active node in the stack, so the context from the beginning of the buffer to
		//the first active node in the stack and in the buffer are written to disk
		else if(mnmin_inmem_level>=0) 
		{
			nwrite_page_no = (mndisk_write_start_pos/mntree_buf_page_size)%mntree_buf_num_of_pages;
			nwrite_page_pos = mndisk_write_start_pos%mntree_buf_page_size;
			ndisk_write_end_pos = mpactive_stack[mnmin_inmem_level].ndisk_pos;

			nfile_pos = ftell(mfpcfp_file);
			if(mndisk_write_start_pos!=nfile_pos)
				fseek(mfpcfp_file, mndisk_write_start_pos-nfile_pos, SEEK_CUR);

			while(mndisk_write_start_pos<ndisk_write_end_pos)
			{
				if(nwrite_page_no==mpactive_stack[mnmin_inmem_level].mem_page_no 
					&& nwrite_page_pos<mpactive_stack[mnmin_inmem_level].mem_page_pos)
				{
					fwrite(&(mptree_buffer[nwrite_page_no][nwrite_page_pos]), sizeof(char), mpactive_stack[mnmin_inmem_level].mem_page_pos-nwrite_page_pos, mfpcfp_file);
					mndisk_write_start_pos += mpactive_stack[mnmin_inmem_level].mem_page_pos-nwrite_page_pos;
					nwrite_page_pos = mpactive_stack[mnmin_inmem_level].mem_page_pos;
					
				}
				else 
				{
					fwrite(&(mptree_buffer[nwrite_page_no][nwrite_page_pos]), sizeof(char), mntree_buf_page_size-nwrite_page_pos, mfpcfp_file);
					mndisk_write_start_pos += (mntree_buf_page_size-nwrite_page_pos);
					nwrite_page_pos = 0;
					nwrite_page_no = (nwrite_page_no+1)%mntree_buf_num_of_pages;					
				}
			}
			if(mndisk_write_start_pos!=ndisk_write_end_pos)
				printf("Error when dumping\n");
			mndisk_start_pos = mndisk_write_start_pos;
			
			top = mnmin_inmem_level;
			while(top<mnactive_top && mndisk_write_start_pos==mpactive_stack[top].ndisk_pos)
			{
				memcpy(&nfill_size, &(mptree_buffer[mpactive_stack[top].mem_page_no][mnpage_size]), sizeof(int));
				nwrite_page_pos += mpactive_stack[top].nfull_node_size;
				if(nwrite_page_pos==mnpage_size)
				{
					nwrite_page_no = (nwrite_page_no+1)%mntree_buf_num_of_pages;
					nwrite_page_pos = 0;
					mndisk_write_start_pos += mpactive_stack[top].nfull_node_size+sizeof(int);
				}
				else if(nwrite_page_pos+nfill_size==mnpage_size)
				{
					nwrite_page_no = (nwrite_page_no+1)%mntree_buf_num_of_pages;
					nwrite_page_pos = 0;
					mndisk_write_start_pos += mpactive_stack[top].nfull_node_size+nfill_size+sizeof(int);
				}
				else
					mndisk_write_start_pos += mpactive_stack[top].nfull_node_size;
				top--;
			}
			if(top>=mnactive_top)
				mndisk_write_start_pos = -1;			
		}
		//No active node in the stack is in the buffer. Dump all the contents in the buffer to disk
		else if(mndisk_write_start_pos!=-1 && mndisk_write_start_pos<mntree_size)
		{
			nwrite_page_no = (mndisk_write_start_pos/mntree_buf_page_size)%mntree_buf_num_of_pages;
			nwrite_page_pos = mndisk_write_start_pos%mntree_buf_page_size;

			nfile_pos = ftell(mfpcfp_file);
			if(mndisk_write_start_pos!=nfile_pos)
				fseek(mfpcfp_file, mndisk_write_start_pos-nfile_pos, SEEK_CUR);

			while(mndisk_write_start_pos!=mntree_size-mncur_page_pos)
			{
				fwrite(&(mptree_buffer[nwrite_page_no][nwrite_page_pos]), sizeof(char), mntree_buf_page_size-nwrite_page_pos, mfpcfp_file);
				mndisk_write_start_pos += mntree_buf_page_size-nwrite_page_pos;
				nwrite_page_pos = 0;
				nwrite_page_no = (nwrite_page_no+1)%mntree_buf_num_of_pages;
			}
			if(mncur_page_pos>0)
			{
				fwrite(mptree_buffer[nwrite_page_no], sizeof(char), mncur_page_pos, mfpcfp_file);
				mndisk_write_start_pos += mncur_page_pos;
			}
			if(mndisk_write_start_pos!=mntree_size)
				printf("Error when dumping\n");
			mndisk_start_pos = mntree_size;
		}
		else 
			printf("Error: no memory can be released\n");
	}

	fflush(mfpcfp_file);
}

//------------------------------------------------------------------------
//Dump all the contents in buffer on disk
//------------------------------------------------------------------------
void CTreeOutBufManager::DumpAll()
{
	int nfill_size, nfile_pos, nwrite_page_no, nwrite_page_pos;

	if(mntree_size==0)
		return;

	if(mndisk_start_pos!=mndisk_write_start_pos)
		printf("Error[DumpAll]: inconsistent start and wirte position: %d %d\n", mndisk_start_pos, mndisk_write_start_pos);

	nfile_pos = ftell(mfpcfp_file);
	if(mndisk_write_start_pos!=nfile_pos)
		fseek(mfpcfp_file, mndisk_write_start_pos-nfile_pos, SEEK_CUR);

	nwrite_page_no = (mndisk_write_start_pos/mntree_buf_page_size)%mntree_buf_num_of_pages;
	nwrite_page_pos = mndisk_write_start_pos%mntree_buf_page_size;

	if(nwrite_page_no!=mncur_page_no)
	{
		fwrite(&(mptree_buffer[nwrite_page_no][nwrite_page_pos]), sizeof(char), mntree_buf_page_size-nwrite_page_pos, mfpcfp_file);
		nwrite_page_no = (nwrite_page_no+1)%mntree_buf_num_of_pages;
		while(nwrite_page_no!=mncur_page_no)
		{
			fwrite(mptree_buffer[nwrite_page_no], sizeof(char), mntree_buf_page_size, mfpcfp_file);
			nwrite_page_no = (nwrite_page_no+1)%mntree_buf_num_of_pages;
		}
		if(mncur_page_pos>0)
		{
			nfill_size = mnpage_size-mncur_page_pos;
			memcpy(&(mptree_buffer[mncur_page_no][mnpage_size]), &nfill_size, sizeof(int));
			fwrite(mptree_buffer[mncur_page_no], sizeof(char), mntree_buf_page_size, mfpcfp_file);
			mntree_size += nfill_size+sizeof(int);
		}
	}
	else
	{
		nfill_size = mnpage_size-mncur_page_pos;
		memcpy(&(mptree_buffer[mncur_page_no][mnpage_size]), &nfill_size, sizeof(int));
		fwrite(&(mptree_buffer[mncur_page_no][nwrite_page_pos]), sizeof(char), mntree_buf_page_size-nwrite_page_pos, mfpcfp_file);
		mntree_size += nfill_size+sizeof(int);
	}

	fflush(mfpcfp_file);
}


//--------------------------------------------------------------------------------------------------
//A CFP-node can either be in active stack, in output buffer or on disk. 
//If a node is in active stack, then its descendant can be in active stack, output buffer or on disk.
//If a node is on disk, then its descendant can be on disk or in output buffer.
//If a node is in output buffer but not in active stack, then all of its desendants are in output buffer.
//---------------------------------------------------------------------------------------------------
bool CTreeOutBufManager::IsOnBorder(int *ppattern, int npat_len, int nsup)
{
	ENTRY *pentries;
	int top, i, nprefix_len, npos;
	bool bfound;

	//return false;

	gnborder_check_times++;

	//if(nsup>=gnmin_sup)
	//	printf("Error: the pattern should not be frequent\n");

	if(npat_len!=mpactive_stack[mnactive_top-1].nprefix_len+1)
		printf("Error: inconsistent pattern length\n");

	for(i=0;i<npat_len;i++)
		mpitem_bitmap[ppattern[i]] = 1;

	bfound = false;

	for(top=mnactive_top-2;top>=0;top--)
	{
		if(mpactive_stack[top].num_of_entries>1)
		{
			pentries = mpactive_stack[top].pentries;
			nprefix_len = mpactive_stack[top].nprefix_len;

			npos = mpactive_stack[top].ncur_order-1;
			while(npos>=0 && mpitem_bitmap[pentries[npos].item]==0)
				npos--;
			if(npos>=0 && pentries[npos].support>=gnmin_sup) // mpitem_bitmap[pentries[npos].item]==1
			{
				if(pentries[npos].child==0)
					printf("Error: the child of the entry should not be NULL\n");
				else if(pentries[npos].child<mndisk_write_start_pos)
					bfound = SearchOnDisk(pentries[npos].child, npat_len-nprefix_len-1);
				else 
					bfound = SearchInBuf(pentries[npos].child, npat_len-nprefix_len-1);
				if(bfound)
					break;
			}
		}
	}
	for(i=0;i<npat_len;i++)
		mpitem_bitmap[ppattern[i]] = 0;

	if(bfound)
		gninfreq_pat_num++;

	return bfound;
}


//-----------------------------------------------------------------------------------------------------------
//Search supersets of ppattern with support nsupport on disk
//-----------------------------------------------------------------------------------------------------------
bool CTreeOutBufManager::SearchOnDisk(int ndisk_pos, int npat_len)
{
	int nfile_pos, num_of_entries, i, j, *pitems;
	int num_of_covered_items;
	bool bfound;
	ENTRY* pentries, cfp_entry;

	nfile_pos = ftell(mfpread_file);
	if(ndisk_pos!=nfile_pos)
		fseek(mfpread_file, ndisk_pos-nfile_pos, SEEK_CUR);
	fread(&num_of_entries, sizeof(int), 1, mfpread_file);

	bfound = false;
	if(num_of_entries==1 || num_of_entries<-1)
	{
		pitems = NULL;
		if(num_of_entries==1)
		{
			fread(&cfp_entry, sizeof(ENTRY)-sizeof(int), 1, mfpread_file);
			pitems = &cfp_entry.item;
			fread(pitems, sizeof(int), 1, mfpread_file);
		}
		else
		{
			num_of_entries = -num_of_entries;
			pitems = NewCFPItemset(num_of_entries);
			fread(&cfp_entry, sizeof(ENTRY)-sizeof(int), 1, mfpread_file);
			fread(pitems, sizeof(int), num_of_entries, mfpread_file);
		}

		if(cfp_entry.support<gnmin_sup || cfp_entry.child==0 && num_of_entries<npat_len)
		{
			if(num_of_entries>1)
				DelCFPItemset(pitems, num_of_entries);
			return false;
		}

		num_of_covered_items = 0;
		for(j=0;j<num_of_entries;j++)
		{
			if(mpitem_bitmap[pitems[j]])
				num_of_covered_items++;
		}

		if(num_of_covered_items==npat_len)
			bfound = true;
		else if(cfp_entry.child!=0)
		{
			if(cfp_entry.child<mndisk_write_start_pos)
				bfound = SearchOnDisk(cfp_entry.child, npat_len-num_of_covered_items);
			else
				bfound = SearchInBuf(cfp_entry.child, npat_len-num_of_covered_items);
		}
		if(num_of_entries>1)
			DelCFPItemset(pitems, num_of_entries);
	}
	else if(num_of_entries>=npat_len)
	{
		pentries = NewEntryArray(num_of_entries);
		fread(pentries, sizeof(ENTRY), num_of_entries, mfpread_file);

		for(i=num_of_entries-1;i>=0;i--)
		{
			if(mpitem_bitmap[pentries[i].item])
				break;
		}
		if(i>=0 && pentries[i].support>=gnmin_sup)
		{
			 if(npat_len==1)
				bfound = true;
			else if(pentries[i].child!=0)
			{
				if(pentries[i].child<mndisk_write_start_pos)
					bfound = SearchOnDisk(pentries[i].child, npat_len-1);
				else
					bfound = SearchInBuf(pentries[i].child, npat_len-1);
			}
		}
		DelEntryArray(pentries, num_of_entries);
	}
	else 
		printf("Error with number of entries\n");

	return bfound;
}


//-----------------------------------------------------------------------------------------------------------
//Search supersets of ppattern with support nsupport in output buffer.
//-----------------------------------------------------------------------------------------------------------
bool CTreeOutBufManager::SearchInBuf(int ndisk_pos, int npat_len)
{
	int npage_pos, npage_no, num_of_entries, i, j, *pitems;
	int num_of_covered_items;
	bool bfound;
	ENTRY* pentries, *pentry;

	npage_no = (ndisk_pos/mntree_buf_page_size)%mntree_buf_num_of_pages;
	npage_pos = ndisk_pos%mntree_buf_page_size;

	memcpy(&num_of_entries, &(mptree_buffer[npage_no][npage_pos]), sizeof(int));
	npage_pos += sizeof(int);

	bfound = false;
	if(num_of_entries==1 || num_of_entries<-1)
	{
		if(num_of_entries==1)
		{
			pentry = (ENTRY*)&(mptree_buffer[npage_no][npage_pos]);
			npage_pos += sizeof(ENTRY)-sizeof(int);
			pitems = (int*)&(mptree_buffer[npage_no][npage_pos]);
		}
		else 
		{
			num_of_entries = - num_of_entries;
			pentry = (ENTRY*)&(mptree_buffer[npage_no][npage_pos]);
			npage_pos += sizeof(ENTRY)-sizeof(int);
			pitems = (int*)&(mptree_buffer[npage_no][npage_pos]);
		}

		if(pentry->support<gnmin_sup || pentry->child==0 && num_of_entries<npat_len)
			return false;

		num_of_covered_items = 0;
		for(j=0;j<num_of_entries;j++)
		{
			if(mpitem_bitmap[pitems[j]])
				num_of_covered_items++;
		}
		if(num_of_covered_items==npat_len)
			bfound = true;
		else if(pentry->child!=0)
			bfound = SearchInBuf(pentry->child, npat_len-num_of_covered_items);

	}
	else if(num_of_entries>=npat_len)
	{
		pentries = (ENTRY*)&(mptree_buffer[npage_no][npage_pos]);

		for(i=num_of_entries-1;i>=0;i--)
		{
			if(mpitem_bitmap[pentries[i].item])
				break;
		}
		if(i>=0 && pentries[i].support>=gnmin_sup)
		{
			if(npat_len==1)
				return true;
			else if(pentries[i].child!=0)
				bfound = SearchInBuf(pentries[i].child, npat_len-1);
		}
	}
	else
		printf("Error with number of entries\n");

	return bfound;
}

//=========================================================
// identify non-closed patterns and mark them
//=========================================================
void CTreeOutBufManager::PruneNonClosed(int *ppattern, int npat_len, int nsupport, int nup_level)
{
	ENTRY *pentries;
	int top, i, nprefix_len, npos;

	gnfreq_pat_num++;
	if(gnmax_pattern_len<npat_len)
		gnmax_pattern_len = npat_len;
	if(gnmax_sup<nsupport)
		gnmax_sup = nsupport;

	for(i=0;i<npat_len;i++)
		mpitem_bitmap[ppattern[i]] = 1;

	for(top=mnactive_top-1-nup_level;top>=0;top--)
	{
		if(mpactive_stack[top].num_of_entries>1)
		{
			pentries = mpactive_stack[top].pentries;
			nprefix_len = mpactive_stack[top].nprefix_len;

			npos = mpactive_stack[top].ncur_order-1;
			while(npos>=0 && mpitem_bitmap[pentries[npos].item]==0)
				npos--;

			if(npos>=0 && (pentries[npos].hash_bitmap & (1<<CFP_HASH_LEN))==0)
			{
				if(pentries[npos].support<nsupport)
					printf("Error: the support of this entry should be larger than %d\n", nsupport);
				else if(pentries[npos].support==nsupport)
					pentries[npos].hash_bitmap |= (1<<CFP_HASH_LEN);
				else if(npat_len-nprefix_len-1>0)
				{
					if(pentries[npos].child!=0)
					{
						if(pentries[npos].child<mndisk_write_start_pos)
							PruneNonClosedOnDisk(pentries[npos].child, npat_len-nprefix_len-1, nsupport);
						else 
							PruneNonClosedInBuf(pentries[npos].child, npat_len-nprefix_len-1, nsupport);
					}
					//else 
					//	printf("Error: the child pointer should not be null\n");
				}
			}
		}
	}
	for(i=0;i<npat_len;i++)
		mpitem_bitmap[ppattern[i]] = 0;

}


//-----------------------------------------------------------------------------------------------------------
//Search supersets of ppattern with support nsupport on disk
//-----------------------------------------------------------------------------------------------------------
void CTreeOutBufManager::PruneNonClosedOnDisk(int ndisk_pos, int npat_len, int nsupport)
{
	int nfile_pos, nwrite_pos, num_of_entries, i, j, *pitems;
	int num_of_covered_items;
	ENTRY* pentries, cfp_entry;

	nfile_pos = ftell(mfpread_file);
	if(ndisk_pos!=nfile_pos)
		fseek(mfpread_file, ndisk_pos-nfile_pos, SEEK_CUR);
	fread(&num_of_entries, sizeof(int), 1, mfpread_file);

	if(num_of_entries==1 || num_of_entries<-1)
	{
		pitems = NULL;
		if(num_of_entries==1)
		{
			fread(&cfp_entry, sizeof(ENTRY)-sizeof(int), 1, mfpread_file);
			pitems = &cfp_entry.item;
			fread(pitems, sizeof(int), 1, mfpread_file);
		}
		else
		{
			num_of_entries = -num_of_entries;
			pitems = NewCFPItemset(num_of_entries);
			fread(&cfp_entry, sizeof(ENTRY)-sizeof(int), 1, mfpread_file);
			fread(pitems, sizeof(int), num_of_entries, mfpread_file);
		}

		if(cfp_entry.hash_bitmap & (1<<CFP_HASH_LEN))
		{
			if(num_of_entries>1)
				DelCFPItemset(pitems, num_of_entries);
			return;
		}

		if(cfp_entry.support<nsupport || cfp_entry.child==0 && num_of_entries<npat_len)
		{
			printf("Error: inconsistent support or items in the CFP-tree node\n");
			if(num_of_entries>1)
				DelCFPItemset(pitems, num_of_entries);
			return;
		}
			
		num_of_covered_items = 0;
		for(j=0;j<num_of_entries;j++)
		{
			if(mpitem_bitmap[pitems[j]])
				num_of_covered_items++;
		}
		if(num_of_covered_items!=npat_len && num_of_covered_items!=num_of_entries)
			printf("Error: the number of matched items should be either equal to pattern length or number of entries\n");

		if(cfp_entry.support==nsupport)
		{
			gnnonclosed_freq_pat_num++;
			cfp_entry.hash_bitmap |= (1<<CFP_HASH_LEN);
			nwrite_pos = ndisk_pos+sizeof(int)+sizeof(int);
			nfile_pos = ftell(mfpcfp_file);
			if(nwrite_pos!=nfile_pos)
				fseek(mfpcfp_file, nwrite_pos-nfile_pos, SEEK_CUR);
			fwrite(&cfp_entry.hash_bitmap, sizeof(int), 1, mfpcfp_file);
		}
		else if(npat_len>num_of_covered_items)
		{
			if(cfp_entry.child!=0)
			{
				if(cfp_entry.child<mndisk_write_start_pos)
					PruneNonClosedOnDisk(cfp_entry.child, npat_len-num_of_covered_items, nsupport);
				else
					PruneNonClosedInBuf(cfp_entry.child, npat_len-num_of_covered_items, nsupport);
			}
			else 
				printf("Error: the child pointer should not be NULL\n");
		}
		if(num_of_entries>1)
			DelCFPItemset(pitems, num_of_entries);
	}
	else if(num_of_entries>=npat_len)
	{
		pentries = NewEntryArray(num_of_entries);
		fread(pentries, sizeof(ENTRY), num_of_entries, mfpread_file);

		for(i=num_of_entries-1;i>=0;i--)
		{
			if(mpitem_bitmap[pentries[i].item])
				break;
		}
		if(i>=0 && (pentries[i].hash_bitmap & (1<<CFP_HASH_LEN))==0)
		{
			if(pentries[i].support<nsupport)
				printf("Error: the support should be no less than %d\n", nsupport);
			else if(pentries[i].support==nsupport)
			{
				gnnonclosed_freq_pat_num++;
				pentries[i].hash_bitmap |= (1<<CFP_HASH_LEN);
				nwrite_pos = ndisk_pos+sizeof(int)+sizeof(ENTRY)*i+sizeof(int);
				nfile_pos = ftell(mfpcfp_file);
				if(nwrite_pos!=nfile_pos)
					fseek(mfpcfp_file, nwrite_pos-nfile_pos, SEEK_CUR);
				fwrite(&pentries[i].hash_bitmap, sizeof(int), 1, mfpcfp_file);
			}
			else if(npat_len>1)
			{
				if(pentries[i].child!=0)
				{
					if(pentries[i].child<mndisk_write_start_pos)
						PruneNonClosedOnDisk(pentries[i].child, npat_len-1, nsupport);
					else
						PruneNonClosedInBuf(pentries[i].child, npat_len-1, nsupport);
				}
				//else
				//	printf("Error: the child pointer should not be null\n");
			}
		}
		DelEntryArray(pentries, num_of_entries);
	}
	else 
		printf("Error with number of entries\n");

}


//-----------------------------------------------------------------------------------------------------------
//Search supersets of ppattern with support nsupport in output buffer.
//-----------------------------------------------------------------------------------------------------------
void CTreeOutBufManager::PruneNonClosedInBuf(int ndisk_pos, int npat_len, int nsupport)
{
	int npage_pos, npage_no, num_of_entries, i, j, *pitems;
	int num_of_covered_items;
	ENTRY* pentries, *pentry;

	npage_no = (ndisk_pos/mntree_buf_page_size)%mntree_buf_num_of_pages;
	npage_pos = ndisk_pos%mntree_buf_page_size;

	memcpy(&num_of_entries, &(mptree_buffer[npage_no][npage_pos]), sizeof(int));
	npage_pos += sizeof(int);

	if(num_of_entries==1 || num_of_entries<-1)
	{
		if(num_of_entries==1)
		{
			pentry = (ENTRY*)&(mptree_buffer[npage_no][npage_pos]);
			npage_pos += sizeof(ENTRY)-sizeof(int);
			pitems = (int*)&(mptree_buffer[npage_no][npage_pos]);
		}
		else 
		{
			num_of_entries = - num_of_entries;
			pentry = (ENTRY*)&(mptree_buffer[npage_no][npage_pos]);
			npage_pos += sizeof(ENTRY)-sizeof(int);
			pitems = (int*)&(mptree_buffer[npage_no][npage_pos]);
		}

		if(pentry->hash_bitmap & (1<<CFP_HASH_LEN))
			return;

		if(pentry->support<nsupport || pentry->child==0 && num_of_entries<npat_len)
		{
			printf("Error: inconsistent support or items in the CFP-tree node\n");
			return;
		}
			
		num_of_covered_items = 0;
		for(j=0;j<num_of_entries;j++)
		{
			if(mpitem_bitmap[pitems[j]])
				num_of_covered_items++;
		}
		if(num_of_covered_items!=npat_len && num_of_covered_items!=num_of_entries)
			printf("Error: the number of matched items should be either equal to pattern length or number of entries\n");

		if(pentry->support==nsupport)
		{
			pentry->hash_bitmap |= (1<<CFP_HASH_LEN);
			gnnonclosed_freq_pat_num++;
		}
		else if(npat_len>num_of_covered_items)
		{
			if(pentry->child!=0)
				PruneNonClosedInBuf(pentry->child, npat_len-num_of_covered_items, nsupport);
			else
				printf("Error: the child pointer should not be null\n");
		}
	}
	else if(num_of_entries>=npat_len)
	{
		pentries = (ENTRY*)&(mptree_buffer[npage_no][npage_pos]);

		for(i=num_of_entries-1;i>=0;i--)
		{
			if(mpitem_bitmap[pentries[i].item])
				break;
		}
		if(i>=0 && (pentries[i].hash_bitmap & (1<<CFP_HASH_LEN))==0)
		{
			if(pentries[i].support<nsupport)
				printf("Error: the support should be no less than %d\n", nsupport);
			else if(pentries[i].support==nsupport)
			{
				pentries[i].hash_bitmap |= (1<<CFP_HASH_LEN);
				gnnonclosed_freq_pat_num++;
			}
			else if(npat_len>1)
			{
				if(pentries[i].child!=0)
					PruneNonClosedInBuf(pentries[i].child, npat_len-1, nsupport);
				//else
				//	printf("Error: the child pointer should not be null\n");
			}
		}
	}
	else
		printf("Error with number of entries\n");
}

int binary_search(ENTRY *pentries, int nlength, int ntgt_item)
{
	int nstart, nend, nmid, i;

	if(nlength<10)
	{
		for(i=nlength-1;i>=0;i--)
		{
			if(pentries[i].item==ntgt_item)
				return i;
			else if(pentries[i].item<ntgt_item)
				return -1;
		}
	}
	else 
	{
		nstart = 0;
		nend = nlength-1;
		while(nend>=nstart)
		{
			nmid = (nstart+nend)/2;
			if(pentries[nmid].item==ntgt_item)
				return nmid;
			else if(pentries[nmid].item<ntgt_item)
				nstart = nmid+1;
			else 
				nend = nmid-1;
		}
	}
	return -1;
}

