#include <math.h>
#include "pvalue.h"

double *gpdfactorials;

int gnpvalue_buf_size;
int gnpvalue_cutoff_sup;
double* gppvalue_buf;
double** gppsup_pvalues;
int *gpsup_filled_pos;


double *gphigh_sup_pvalues; 
int gnbuffered_high_sup;
int gnhigh_sup_filled_pos;

int gnunbuffered_pvalues;
int gnpartially_buffered_pvalues;

double *gptgt_values;
int *gpint_tgt_values;
GROUP_TGT_VALUE* gpgroup_tgt_values;
double *gpgroup_rank_avgs;


DYN_PVALUE_BUF *gpdyn_pvalue_bufs;


// -------------------------  t-test  ------------------------------------
double gppvalues[20] = {1.0, 0.5, 0.4, 0.3, 0.2, 0.1, 0.05, 0.02, 0.01, 0.002, 
                        0.001, 0.0001, 0.00001, 0.000001, 0.0000001, 0.00000001, 0.000000001, 0.0000000001, 0.00000000001, 0.000000000001};

double gpdf1_t_table[20] = {0.000, 1.000, 1.376, 1.963, 3.078, 6.314, 12.71, 31.82, 63.66, 318.31,   636.62, 6366, 63662, 636622, 6366405, 63682668, 638694827};
double gpdf2_t_table[20] = {0.000, 0.816, 1.061, 1.386, 1.886, 2.920, 4.303, 6.965, 9.925, 22.327,   31.599, 99.99, 316.2, 1000, 3162, 10000, 31623, 99999, 316228, 999955};
double gpdf3_t_table[20] = {0.000, 0.765, 0.978, 1.250, 1.638, 2.353, 3.182, 4.541, 5.841, 10.215,   12.924, 28.00, 60.40, 130.2, 280.4, 604.2, 1302, 2804, 6042, 13016};
double gpdf4_t_table[20] = {0.000, 0.741, 0.941, 1.190, 1.533, 2.132, 2.776, 3.747, 4.604, 7.173,     8.610, 15.54, 27.77, 49.46, 87.99, 156.5, 278.3, 494.9, 880.1, 1565};
double gpdf5_t_table[20] = {0.000, 0.727, 0.920, 1.156, 1.476, 2.015, 2.571, 3.365, 4.032, 5.893,     6.869, 11.18, 17.90, 28.48, 45.21, 71.69, 113.7, 180.1, 285.5, 452.5};
double gpdf6_t_table[20] = {0.000, 0.718, 0.906, 1.134, 1.440, 1.943, 2.447, 3.143, 3.707, 5.208,     5.959, 9.082, 13.56, 20.05, 29.53, 43.41, 63.77, 93.63, 137.5, 201.8};
double gpdf7_t_table[20] = {0.000, 0.711, 0.896, 1.119, 1.415, 1.895, 2.365, 2.998, 3.499, 4.785,     5.408, 7.885, 11.21, 15.77, 22.04, 30.72, 42.75, 59.45, 82.64, 114.9};
double gpdf8_t_table[20] = {0.000, 0.706, 0.889, 1.108, 1.397, 1.860, 2.306, 2.896, 3.355, 4.501,     5.041, 7.120, 9.783, 13.26, 17.84, 23.90, 31.96, 42.69, 56.97, 76.01};
double gpdf9_t_table[20] = {0.000, 0.703, 0.883, 1.100, 1.383, 1.833, 2.262, 2.821, 3.250, 4.297,     4.781, 6.594, 8.827, 11.64, 15.21, 19.78, 25.66, 33.22, 42.97, 55.54};
double gpdf10_t_table[20] = {0.000, 0.700, 0.879, 1.093, 1.372, 1.812, 2.228, 2.764, 3.169, 4.144,    4.587, 6.211, 8.150, 10.52, 13.44, 17.08, 21.62, 27.32, 34.47, 43.46};
double gpdf11_t_table[20] = {0.000, 0.697, 0.876, 1.088, 1.363, 1.796, 2.201, 2.718, 3.106, 4.025,    4.437, 5.921, 7.648, 9.701, 12.18, 15.19, 18.86, 23.37, 28.90, 35.71};
double gpdf12_t_table[20] = {0.000, 0.695, 0.873, 1.083, 1.356, 1.782, 2.179, 2.681, 3.055, 3.930,    4.318, 5.694, 7.261, 9.085, 11.24, 13.81, 16.88, 20.58, 25.03, 30.42};
double gpdf13_t_table[20] = {0.000, 0.694, 0.870, 1.079, 1.350, 1.771, 2.160, 2.650, 3.012, 3.852,    4.221, 5.513, 6.954, 8.603, 10.52, 12.76, 15.40, 18.52, 22.22, 26.63};
double gpdf14_t_table[20] = {0.000, 0.692, 0.868, 1.076, 1.345, 1.761, 2.145, 2.624, 2.977, 3.787,    4.140, 5.363, 6.706, 8.218, 9.945, 11.94, 14.25, 16.95, 20.11, 23.81};
double gpdf15_t_table[20] = {0.000, 0.691, 0.866, 1.074, 1.341, 1.753, 2.131, 2.602, 2.947, 3.733,    4.073, 5.239, 6.502, 7.903, 9.483, 11.28, 13.34, 15.72, 18.47, 21.65};
double gpdf16_t_table[20] = {0.000, 0.690, 0.865, 1.071, 1.337, 1.746, 2.120, 2.583, 2.921, 3.686,    4.015, 5.134, 6.330, 7.642, 9.102, 10.75, 12.61, 14.73, 17.16, 19.94};
double gpdf17_t_table[20] = {0.000, 0.689, 0.863, 1.069, 1.333, 1.740, 2.110, 2.567, 2.898, 3.646,    3.965, 5.044, 6.184, 7.421, 8.784, 10.30, 12.00, 13.93, 16.10, 18.57};
double gpdf18_t_table[20] = {0.000, 0.688, 0.862, 1.067, 1.330, 1.734, 2.101, 2.552, 2.878, 3.610,    3.922, 4.966, 6.059, 7.232, 8.513, 9.927, 11.50, 13.26, 15.23, 17.45};
double gpdf19_t_table[20] = {0.000, 0.688, 0.861, 1.066, 1.328, 1.729, 2.093, 2.539, 2.861, 3.579,    3.883, 4.897, 5.949, 7.069, 8.281, 9.607, 11.07, 12.69, 14.50, 16.52};
double gpdf20_t_table[20] = {0.000, 0.687, 0.860, 1.064, 1.325, 1.725, 2.086, 2.528, 2.845, 3.552,    3.850, 4.837, 5.854, 6.927, 8.080, 9.331, 10.70, 12.21, 13.88, 15.73};
double gpdf21_t_table[20] = {0.000, 0.686, 0.859, 1.063, 1.323, 1.721, 2.080, 2.518, 2.831, 3.527,    3.819, 4.784, 5.769, 6.802, 7.904, 9.091, 10.38, 11.80, 13.35, 15.06};
double gpdf22_t_table[20] = {0.000, 0.686, 0.858, 1.061, 1.321, 1.717, 2.074, 2.508, 2.819, 3.505,    3.792, 4.736, 5.694, 6.691, 7.748, 8.881, 10.10, 11.43, 12.89, 14.48};
double gpdf23_t_table[20] = {0.000, 0.685, 0.858, 1.060, 1.319, 1.714, 2.069, 2.500, 2.807, 3.485,    3.768, 4.693, 5.626, 6.593, 7.610, 8.694, 9.859, 11.12, 12.48, 13.97};
double gpdf24_t_table[20] = {0.000, 0.685, 0.857, 1.059, 1.318, 1.711, 2.064, 2.492, 2.797, 3.467,    3.745, 4.654, 5.566, 6.504, 7.487, 8.528, 9.641, 10.84, 12.13, 13.53};
double gpdf25_t_table[20] = {0.000, 0.684, 0.856, 1.058, 1.316, 1.708, 2.060, 2.485, 2.787, 3.450,    3.725, 4.619, 5.511, 6.424, 7.376, 8.379, 9.446, 10.59, 11.82, 13.14};
double gpdf26_t_table[20] = {0.000, 0.684, 0.856, 1.058, 1.315, 1.706, 2.056, 2.479, 2.779, 3.435,    3.707, 4.587, 5.461, 6.352, 7.276, 8.245, 9.272, 10.37, 11.54, 12.79};
double gpdf27_t_table[20] = {0.000, 0.684, 0.855, 1.057, 1.314, 1.703, 2.052, 2.473, 2.771, 3.421,    3.690, 4.558, 5.415, 6.286, 7.185, 8.124, 9.114, 10.16, 11.28, 12.48};
double gpdf28_t_table[20] = {0.000, 0.683, 0.855, 1.056, 1.313, 1.701, 2.048, 2.467, 2.763, 3.408,    3.674, 4.530, 5.373, 6.225, 7.102, 8.014, 8.971, 9.983, 11.06, 12.20};
double gpdf29_t_table[20] = {0.000, 0.683, 0.854, 1.055, 1.311, 1.699, 2.045, 2.462, 2.756, 3.396,    3.659, 4.506, 5.335, 6.170, 7.026, 7.913, 8.841, 9.818, 10.85, 11.95};
double gpdf30_t_table[20] = {0.000, 0.683, 0.854, 1.055, 1.310, 1.697, 2.042, 2.457, 2.750, 3.385,    3.646, 4.482, 5.299, 6.119, 6.956, 7.820, 8.722, 9.668, 10.67, 11.72};
double gpdf40_t_table[20] = {0.000, 0.681, 0.851, 1.050, 1.303, 1.684, 2.021, 2.423, 2.704, 3.307,    3.551, 4.321, 5.053, 5.768, 6.480, 7.197, 7.925, 8.670, 9.437, 10.23};
double gpdf60_t_table[20] = {0.000, 0.679, 0.848, 1.045, 1.296, 1.671, 2.000, 2.390, 2.660, 3.232,    3.460, 4.169, 4.825, 5.449, 6.054, 6.646, 7.233, 7.819, 8.406, 8.997};
double gpdf80_t_table[20] = {0.000, 0.678, 0.846, 1.043, 1.292, 1.664, 1.990, 2.374, 2.639, 3.195,    3.416, 4.096, 4.717, 5.300, 5.857, 6.396, 6.922, 7.441, 7.953, 8.463};
double gpdf100_t_table[20] = {0.000, 0.677, 0.845, 1.042, 1.290, 1.660, 1.984, 2.364, 2.626, 3.174,   3.390, 4.053, 4.654, 5.214, 5.744, 6.253, 6.746, 7.227, 7.700, 8.166};
double gpdf1000_t_table[20] = {0.000, 0.675, 0.842, 1.037, 1.282, 1.646, 1.962, 2.330, 2.581, 3.098,  3.300, 3.906, 4.440, 4.922, 5.366, 5.780, 6.169, 6.537, 6.888, 7.224};

double *gppt_table[30] = {NULL, gpdf1_t_table, gpdf2_t_table, gpdf3_t_table, gpdf4_t_table, gpdf5_t_table,
						gpdf6_t_table, gpdf7_t_table, gpdf8_t_table, gpdf9_t_table, gpdf10_t_table, gpdf11_t_table,
						gpdf12_t_table, gpdf13_t_table, gpdf14_t_table, gpdf15_t_table, gpdf16_t_table, gpdf17_t_table,
						gpdf18_t_table, gpdf19_t_table, gpdf20_t_table, gpdf21_t_table, gpdf22_t_table, gpdf23_t_table,
						gpdf24_t_table, gpdf25_t_table, gpdf26_t_table, gpdf27_t_table, gpdf28_t_table, gpdf29_t_table};




// ----------------------- x2-test ---------------------------------------
double gpdf1_x2_table[20]  = {0.000, 0.455, 0.708, 1.074, 1.642,   2.706, 3.841, 5.412, 6.635, 9.550,    10.83, 15.14, 19.51, 23.93, 28.37,    32.84, 37.32, 41.82, 46.33, 50.84};
double gpdf2_x2_table[20]  = {0.000, 1.386, 1.833, 2.408, 3.219,   4.605, 5.991, 7.824, 9.210, 12.43,    13.82, 18.42, 23.03, 27.63, 32.24,    36.84, 41.45, 46.05, 50.66, 55.26};
double gpdf3_x2_table[20]  = {0.000, 2.366, 2.946, 3.665, 4.642,   6.251, 7.815, 9.837, 11.34, 14.80,    16.27, 21.11, 25.90, 30.66, 35.41,    40.13, 44.84, 49.54, 54.23, 58.92};
double gpdf4_x2_table[20]  = {0.000, 3.357, 4.045, 4.878, 5.989,   7.779, 9.488, 11.67, 13.28, 16.92,    18.47, 23.51, 28.47, 33.38, 38.24,    43.07, 47.88, 52.67, 57.44, 62.20};
double gpdf5_x2_table[20]  = {0.000, 4.351, 5.132, 6.064, 7.289,   9.236, 11.07, 13.39, 15.09, 18.91,    20.52, 25.74, 30.86, 35.89, 40.86,    45.79, 50.69, 55.56, 60.41, 65.24};
double gpdf6_x2_table[20]  = {0.000, 5.348, 6.211, 7.231, 8.558,   10.64, 12.59, 15.03, 16.81, 20.79,    22.46, 27.86, 33.11, 38.26, 43.34,    48.36, 53.34, 58.29, 63.21, 68.10};
double gpdf7_x2_table[20]  = {0.000, 6.346, 7.283, 8.383, 9.803,   12.02, 14.07, 16.62, 18.48, 22.60,    24.32, 29.88, 35.26, 40.52, 45.70,    50.81, 55.87, 60.90, 65.88, 70.84};
double gpdf8_x2_table[20]  = {0.000, 7.344, 8.351, 9.524, 11.03,   13.36, 15.51, 18.17, 20.09, 24.35,    26.12, 31.83, 37.33, 42.70, 47.97,    53.17, 58.31, 63.40, 68.45, 73.47};
double gpdf9_x2_table[20]  = {0.000, 8.343, 9.414, 10.66, 12.24,   14.68, 16.92, 19.68, 21.67, 26.06,    27.88, 33.72, 39.34, 44.81, 50.17,    55.45, 60.66, 65.82, 70.93, 76.01};
double gpdf10_x2_table[20] = {0.000, 9.342, 10.47, 11.78, 13.44,   15.99, 18.31, 21.16, 23.21, 27.72,    29.59, 35.56, 41.30, 46.86, 52.31,    57.66, 62.95, 68.17, 73.34, 78.47};
double gpdf11_x2_table[20] = {0.000, 10.34, 11.53, 12.90, 14.63,   17.28, 19.68, 22.62, 24.72, 29.35,    31.26, 37.37, 43.21, 48.87, 54.39,    59.82, 65.17, 70.46, 75.69, 80.87};
double gpdf12_x2_table[20] = {0.000, 11.34, 12.58, 14.01, 15.81,   18.55, 21.03, 24.05, 26.22, 30.96,    32.91, 39.13, 45.08, 50.83, 56.43,    61.93, 67.35, 72.69, 77.98, 83.22};
double gpdf20_x2_table[20]   = {0.000, 19.34, 20.95, 22.77, 25.04,   28.41, 31.41, 35.02, 37.57, 43.07,   45.31, 52.39, 59.04, 65.42, 71.59,    77.60, 83.48, 89.26, 94.94, 100.6};
double gpdf30_x2_table[20]   = {0.000, 29.34, 31.32, 33.53, 36.25,   40.26, 43.77, 47.96, 50.89, 57.17,   59.70, 67.63, 75.02, 82.04, 88.79,    95.33, 101.7, 107.9, 114.0, 120.1};
double gpdf40_x2_table[20]   = {0.000, 39.34, 41.62, 44.16, 47.27,   51.81, 55.76, 60.44, 63.69, 70.62,   73.40, 82.06, 90.08, 97.65, 104.9,    111.9, 118.7, 125.3, 131.8, 138.1};
double gpdf60_x2_table[20]   = {0.000, 59.33, 62.13, 65.23, 68.97,   74.40, 79.08, 84.58, 88.38, 96.40,   99.61, 109.5, 118.6, 127.1, 135.2,    143.0, 150.5, 157.8, 164.9, 171.9};
double gpdf80_x2_table[20]   = {0.000, 79.33, 82.57, 86.12, 90.41,   96.58, 101.9, 108.1, 112.3, 121.3,   124.8, 135.8, 145.8, 155.1, 163.9,    172.3, 180.5, 188.4, 196.0, 203.5};
double gpdf100_x2_table[20]  = {0.000, 99.33, 102.9, 106.9, 111.7,   118.5, 124.3, 131.1, 135.8, 145.6,   149.4, 161.3, 172.1, 182.1, 191.6,    200.6, 209.3, 217.7, 225.9, 233.8};
double gpdf1000_x2_table[20] = {0.000, 999.3, 1011, 1023, 1037,   1058, 1075, 1094, 1107, 1134,    1144, 1175, 1202, 1227, 1250,   1272, 1292, 1311,  1330, 1348};

double *gppx2_table[13] = {NULL, gpdf1_x2_table, gpdf2_x2_table, gpdf3_x2_table, gpdf4_x2_table, gpdf5_x2_table,
						gpdf6_x2_table, gpdf7_x2_table, gpdf8_x2_table, gpdf9_x2_table, gpdf10_x2_table, gpdf11_x2_table,
						gpdf12_x2_table};


//------------------------ fisher's exact test ------------------------------
void InitFactorials(double* &pdfactorials, int n)
{
	int i, nbuf_pos, nbuffered_sup_num;

	pdfactorials = NewDoubleArray(n+1);

	pdfactorials[0] = 0;
	pdfactorials[1] = 0;

	for(i=2;i<=n;i++)
		pdfactorials[i] = pdfactorials[i-1]+log10((double)i);


	if(gnPVALUE_BUF_SIZE>0)//create buffer for storing p-values of association rules
	{
		if(gbrule_multiclass_pairwise)
		{
			gpdyn_pvalue_bufs = new DYN_PVALUE_BUF[gnum_of_tgt_values];
			IncMemSize(sizeof(DYN_PVALUE_BUF)*gnum_of_tgt_values);
			for(i=0;i<gnum_of_tgt_values;i++)
			{
				gpdyn_pvalue_bufs[i].ppvalues = NewDoubleArray(gnmax_sup+1);
				gpdyn_pvalue_bufs[i].nsup = -1;
				gpdyn_pvalue_bufs[i].nfilled_pos = gnmax_sup+1;
			}
		}
		else
		{
			printf("Buffer size for p-values: %.3f\n", (double)gnPVALUE_BUF_SIZE/(1<<20));

			gphigh_sup_pvalues = NewDoubleArray(gnmax_sup+1);
			gnbuffered_high_sup = -1;
			gnhigh_sup_filled_pos = gnmax_sup+1;

			gnpvalue_buf_size = 0;
			for(i=gnmin_sup;i<=gnmax_sup && gnmax_sup+gnpvalue_buf_size+i+1<=(int)(gnPVALUE_BUF_SIZE/sizeof(double));i++)
				gnpvalue_buf_size += (i+1);
			gnpvalue_cutoff_sup = i;

			nbuffered_sup_num = gnpvalue_cutoff_sup-gnmin_sup;

			gppvalue_buf = NewDoubleArray(gnpvalue_buf_size);
			gppsup_pvalues = new double*[nbuffered_sup_num];
			IncMemSize(sizeof(double*)*nbuffered_sup_num);
			gpsup_filled_pos = NewIntArray(nbuffered_sup_num);
			
			nbuf_pos = 0;
			for(i=0;i<nbuffered_sup_num;i++)
			{
				gppsup_pvalues[i] = &gppvalue_buf[nbuf_pos];
				nbuf_pos += gnmin_sup+i+1;
				gpsup_filled_pos[i] = gnmin_sup+i+1;
			}

			if(nbuf_pos!=gnpvalue_buf_size)
				printf("Error: inconsistent buffer size\n");

			printf("maximal support of rules: %d\n", gnmax_sup);
			printf("P-values of rules with coverage between %d and %d are buffered\n\n", gnmin_sup, gnpvalue_cutoff_sup);

		}
	}
}

void DelFactorials(double *pdfactorials, int n)
{
	DelDoubleArray(pdfactorials, n+1);

	if(gnPVALUE_BUF_SIZE>0)
	{
		if(gbrule_multiclass_pairwise)
		{
			for(int i=0;i<gnum_of_tgt_values;i++)
				DelDoubleArray(gpdyn_pvalue_bufs[i].ppvalues, gnmax_sup+1);
			delete []gpdyn_pvalue_bufs;
			DecMemSize(sizeof(DYN_PVALUE_BUF)*gnum_of_tgt_values);
		}
		else
		{
			delete []gppsup_pvalues;
			DecMemSize(sizeof(double*)*(gnpvalue_cutoff_sup-gnmin_sup));
			DelIntArray(gpsup_filled_pos, gnpvalue_cutoff_sup-gnmin_sup);
			DelDoubleArray(gppvalue_buf, gnpvalue_buf_size);
			DelDoubleArray(gphigh_sup_pvalues, gnmax_sup+1);
		}
	}
}

double HyperGeometric(int n, int x, int y, int k)
{
	double dpvalue;

	dpvalue = gpdfactorials[x]+gpdfactorials[y]+gpdfactorials[n-x]+gpdfactorials[n-y];
	dpvalue -= (gpdfactorials[n]+gpdfactorials[k]+gpdfactorials[x-k]+gpdfactorials[y-k]+gpdfactorials[n-x-y+k]);

	dpvalue = pow(10, dpvalue);

	return dpvalue;
}

double CalcFisherPvalue(int n, int x, int y, int k)
{
	int i;
	double dpvalue;

	if(k==0)
		return 1;
	else
	{
		dpvalue = 0;
		i = y;
		if(i>x)
			i = x;
		while(i>=k)
		{
			dpvalue += HyperGeometric(n, x, y, i);
			i--;
		}	
	}

	return dpvalue;
}

double CalcFisherPvalue(int y, int k)
{
	int i, x;
	double dpvalue;

	x = ((int*)gptgt_stat_array)[0];

	if(gnPVALUE_BUF_SIZE==0)
		CalcFisherPvalue(gndb_size, x, y, k);
	else if(k==0)
		return 1;
	else if(y>=gnpvalue_cutoff_sup) //the coverage of the rule is too high to be put in the buffer, so the p-values are temporarily buffered
	{
		if(y!=gnbuffered_high_sup)
		{
			gnunbuffered_pvalues++;

			gnbuffered_high_sup = y;

			dpvalue = 0;
			i = y;
			if(i>x)
				i = x;
			while(i>=k)
			{
				dpvalue += HyperGeometric(gndb_size, x, y, i);
				gphigh_sup_pvalues[i] = dpvalue;
				i--;
			}
			gnhigh_sup_filled_pos = k;
		}
		else if(k<gnhigh_sup_filled_pos)
		{
			gnpartially_buffered_pvalues++;
			dpvalue = gphigh_sup_pvalues[gnhigh_sup_filled_pos];
			i = gnhigh_sup_filled_pos-1;

			while(i>=k)
			{
				dpvalue += HyperGeometric(gndb_size, x, y, i);
				gphigh_sup_pvalues[i] = dpvalue;
				i--;
			}
			gnhigh_sup_filled_pos = k;
		}
		else
			dpvalue = gphigh_sup_pvalues[k];
	}
	else if(k>=gpsup_filled_pos[y-gnmin_sup])//completedly buffered
	{
		dpvalue = gppsup_pvalues[y-gnmin_sup][k];
	}
	else 
	{
		if(gpsup_filled_pos[y-gnmin_sup]==y+1) //not buffered
		{
			gnunbuffered_pvalues++;
			dpvalue = 0;
			i = y;
			if(i>x)
				i = x;
		}
		else //paritially buffered
		{
			gnpartially_buffered_pvalues++;
			dpvalue = gppsup_pvalues[y-gnmin_sup][gpsup_filled_pos[y-gnmin_sup]];
			i = gpsup_filled_pos[y-gnmin_sup]-1;
			
		}

		while(i>=k)
		{
			dpvalue += HyperGeometric(gndb_size, x, y, i);
			gppsup_pvalues[y-gnmin_sup][i] = dpvalue;
			i--;
		}
		gpsup_filled_pos[y-gnmin_sup] = k;
	}

	//double dpvalue1 = CalcFisherPvalue(gndb_size, ((int*)gptgt_stat_array)[0], y, k);
	//if(dpvalue1!=dpvalue)
	//	printf("Error: inconsistent p-value\n");

	return dpvalue;
}

double CalcTwoTailedFisherPvalue(int n, int x, int y, int k)
{
	int i;
	double dorig_pvalue, dpvalue_sum, dpvalue;

	dorig_pvalue = HyperGeometric(n, x, y, k);
	dpvalue_sum = dorig_pvalue;

	i = y-(n-x);
	if(i<0)
		i = 0;
	while(i<k)
	{
		dpvalue = HyperGeometric(n, x, y, i);
		if(dpvalue<=dorig_pvalue)
			dpvalue_sum += dpvalue;
		else 
			break;
		i++;
	}
	i = x;
	if(i>y)
		i = y;
	while(i>k)
	{
		dpvalue = HyperGeometric(n, x, y, i);
		if(dpvalue<=dorig_pvalue)
			dpvalue_sum += dpvalue;
		else 
			break;
		i--;
	}

	return dpvalue_sum;
}

int FillTwoTailedPvalues(double *ppvalues, int x, int y)
{
	int nupper_bound, nlower_bound, i, u, l;
	double dpvalue;

	nupper_bound = y;
	if(nupper_bound>x)
		nupper_bound = x;
	nlower_bound = y-(gndb_size-x);
	if(nlower_bound<0)
		nlower_bound = 0;

	for(i=nlower_bound;i<=nupper_bound;i++)
		ppvalues[i] = HyperGeometric(gndb_size, x, y, i);

	l = nlower_bound;
	u = nupper_bound;
	dpvalue = 0;
	while(l<=u)
	{
		if(ppvalues[l]<=ppvalues[u])
		{
			while(l<u && ppvalues[l]<=ppvalues[u])
			{
				dpvalue += ppvalues[l];
				ppvalues[l] = dpvalue;
				l++;
			}
			dpvalue += ppvalues[u];
			ppvalues[u] = dpvalue;
			u--;
		}
		else 
		{
			while(l<u && ppvalues[l]>ppvalues[u])
			{
				dpvalue += ppvalues[u];
				ppvalues[u] = dpvalue;
				u--;
			}
			dpvalue += ppvalues[l];
			ppvalues[l] = dpvalue;
			l++;
		}
	}

	if(u>=nlower_bound && ppvalues[u]>=ppvalues[l])
		return u;
	else
		return l;
}

double CalcTwoTailedFisherPvalue(int y, int k)
{
	int x; 
	double dpvalue;

	x = ((int*)gptgt_stat_array)[0];

	if(gnPVALUE_BUF_SIZE==0)
		dpvalue = CalcTwoTailedFisherPvalue(gndb_size, x, y, k);
	else if(y>=gnpvalue_cutoff_sup) 
	{
		if(y!=gnbuffered_high_sup)
		{
			gnunbuffered_pvalues++;

			gnbuffered_high_sup = y;
			gnhigh_sup_filled_pos = FillTwoTailedPvalues(gphigh_sup_pvalues, x, y);

			dpvalue = gphigh_sup_pvalues[k];
		}
		else
			dpvalue = gphigh_sup_pvalues[k];
	}
	else if(gpsup_filled_pos[y-gnmin_sup]==y+1) //not buffered
	{
		gnunbuffered_pvalues++;

		gpsup_filled_pos[y-gnmin_sup] = FillTwoTailedPvalues(gppsup_pvalues[y-gnmin_sup], x, y);

		dpvalue = gppsup_pvalues[y-gnmin_sup][k];
	}
	else
		dpvalue = gppsup_pvalues[y-gnmin_sup][k];

	//double dpvalue1 = CalcTwoTailedFisherPvalue(gndb_size, ((int*)gptgt_stat_array)[0], y, k);
	//if(dpvalue1!=dpvalue)
	//	printf("Error: inconsistent p-value\n");

	return dpvalue;
}

double CalcTwoTailedFisherPvalue(int y, int k, int nclass_no)
{
	int x; 
	double dpvalue;

	x = ((int*)gptgt_stat_array)[nclass_no];

	if(gnPVALUE_BUF_SIZE==0)
		dpvalue = CalcTwoTailedFisherPvalue(gndb_size, x, y, k);
	else
	{
		if(y!=gpdyn_pvalue_bufs[nclass_no].nsup)
		{
			gnunbuffered_pvalues++;

			gpdyn_pvalue_bufs[nclass_no].nsup = y;
			gpdyn_pvalue_bufs[nclass_no].nfilled_pos = FillTwoTailedPvalues(gpdyn_pvalue_bufs[nclass_no].ppvalues, x, y);

			dpvalue = gpdyn_pvalue_bufs[nclass_no].ppvalues[k];
		}
		else
			dpvalue = gpdyn_pvalue_bufs[nclass_no].ppvalues[k];
	}

	//double dpvalue1 = CalcTwoTailedFisherPvalue(gndb_size, x, y, k);
	//if(dpvalue1!=dpvalue)
	//	printf("Error: inconsistent p-value\n");

	return dpvalue;
}


double CalcANOVA(double dFscore, int d1, int d2)
{
	double x, dtemp, dpvalue;
	int a, b, i;

	x = d1*dFscore/(d1*dFscore+d2);
	a = d1/2;
	b = d2/2;

	dpvalue = 0;
	for(i=a;i<a+b;i++)
	{
		dtemp = gpdfactorials[a+b-1]-gpdfactorials[i]-gpdfactorials[a+b-1-i];
		dtemp += i*log10(x)+(a+b-1-i)*log10(1-x);
		dpvalue += pow(10, dtemp);
	}

	return dpvalue;
}


int comp_group_tgt_value(const void* e1, const void*e2)
{
	GROUP_TGT_VALUE *p1, *p2;

	p1 = (GROUP_TGT_VALUE*)e1;
	p2 = (GROUP_TGT_VALUE*)e2;
	if(p1->dtgt_value < p2->dtgt_value)
		return -1;
	else if(p1->dtgt_value > p2->dtgt_value)
		return 1;
	else
		return 0;
}

// --------------  calculate p-value for pairwise comparison -------------------
double CalcPvalue(DIFF_ITEM *pdiff_item1, DIFF_ITEM *pdiff_item2, double &dscore, bool bhas_min_diff)
{
	double dmean_diff, dpvalue;
	int i;

	if(gntgt_attr_type==NOMINAL || gntgt_attr_type==ORDINAL && (gsztarget_value[0]!=0 || gnum_of_tgt_values==2))
	{
		double dtotal, dclass_sum, dexpected, ddiff;

		if(gsztarget_value[0]!=0)
		{
			dmean_diff = pdiff_item1->dmean-pdiff_item2->dmean;
			if(dmean_diff<0)
				dmean_diff = -dmean_diff;

			if(dmean_diff>0 && (!bhas_min_diff || dmean_diff>=gdmin_diff))
			{
				if(gntest_statisitic_method==X2_TEST)
				{
					dtotal = pdiff_item1->nsupport+pdiff_item2->nsupport;
					dclass_sum = ((int*)pdiff_item1->ptgt_stat)[0]+((int*)pdiff_item2->ptgt_stat)[0];
					
					dexpected = pdiff_item1->nsupport*dclass_sum/dtotal;
					ddiff = abs(((int*)pdiff_item1->ptgt_stat)[0]-dexpected)-0.5;
					if(dexpected>0)
						dscore = ddiff*ddiff/dexpected;

					dexpected = pdiff_item2->nsupport*dclass_sum/dtotal;
					ddiff = abs(((int*)pdiff_item2->ptgt_stat)[0]-dexpected)-0.5;
					if(dexpected>0)
						dscore += ddiff*ddiff/dexpected;

					dclass_sum = dtotal-dclass_sum;
					dexpected = pdiff_item1->nsupport*dclass_sum/dtotal;
					ddiff = abs(pdiff_item1->nsupport-((int*)pdiff_item1->ptgt_stat)[0]-dexpected)-0.5;
					if(dexpected>0)
						dscore += ddiff*ddiff/dexpected;

					dexpected = pdiff_item2->nsupport*dclass_sum/dtotal;
					ddiff = abs(pdiff_item2->nsupport-((int*)pdiff_item2->ptgt_stat)[0]-dexpected)-0.5;
					if(dexpected>0)
						dscore += ddiff*ddiff/dexpected;

					dpvalue = LookupX2table(dscore, 1);
				}
				else 
				{
					dpvalue = CalcTwoTailedFisherPvalue(pdiff_item1->nsupport+pdiff_item2->nsupport, pdiff_item1->nsupport, ((int*)pdiff_item1->ptgt_stat)[0]+((int*)pdiff_item2->ptgt_stat)[0], ((int*)pdiff_item1->ptgt_stat)[0]);
					dscore = 0;
				}
			}
			else
			{
				dscore = 0;
				dpvalue = 1;
			}
		}
		else 
		{
			if(gnum_of_tgt_values==2 && gntest_statisitic_method==FISHER_EXACT_TEST)
			{
				dpvalue = CalcTwoTailedFisherPvalue(pdiff_item1->nsupport+pdiff_item2->nsupport, pdiff_item1->nsupport, ((int*)pdiff_item1->ptgt_stat)[0]+((int*)pdiff_item2->ptgt_stat)[0], ((int*)pdiff_item1->ptgt_stat)[0]);
				dscore = 0;
			}
			else
			{
				dtotal = pdiff_item1->nsupport+pdiff_item2->nsupport;
				dscore = 0;
				for(i=0;i<gnum_of_tgt_values;i++)
				{
					dclass_sum = ((int*)pdiff_item1->ptgt_stat)[i]+((int*)pdiff_item2->ptgt_stat)[i];
					
					dexpected = pdiff_item1->nsupport*dclass_sum/dtotal;
					ddiff = abs(((int*)pdiff_item1->ptgt_stat)[i]-dexpected)-0.5;
					if(dexpected>0)
						dscore += ddiff*ddiff/dexpected;

					dexpected = pdiff_item2->nsupport*dclass_sum/dtotal;
					ddiff = abs(((int*)pdiff_item2->ptgt_stat)[i]-dexpected)-0.5;
					if(dexpected>0)
						dscore += ddiff*ddiff/dexpected;
				}
				dpvalue = LookupX2table(dscore, gnum_of_tgt_values-1);
			}
		}
	}
	else if(gntgt_attr_type==CONTINUOUS_NORMAL || gntgt_attr_type==CONTINUOUS && pdiff_item1->nsupport>=CLT_THRESHOLD && pdiff_item2->nsupport>=CLT_THRESHOLD)
	{
		//t-test
		double dvariance1, dvariance2, dtemp; 

		dmean_diff = pdiff_item1->dmean-pdiff_item2->dmean;
		if(dmean_diff<0)
			dmean_diff = -dmean_diff;
		if(!bhas_min_diff || dmean_diff>=gdmin_diff)
		{
			dvariance1 = ((TGT_SUM*)pdiff_item1->ptgt_stat)->dsquare_sum-((TGT_SUM*)pdiff_item1->ptgt_stat)->dsum*pdiff_item1->dmean;
			dvariance2 = ((TGT_SUM*)pdiff_item2->ptgt_stat)->dsquare_sum-((TGT_SUM*)pdiff_item2->ptgt_stat)->dsum*pdiff_item2->dmean;
			dtemp = ((dvariance1+dvariance2)/(pdiff_item1->nsupport+pdiff_item2->nsupport-2));
			dtemp *= (1/(double)pdiff_item1->nsupport + 1/(double)pdiff_item2->nsupport);
			dscore = dmean_diff/sqrt(dtemp);

			dpvalue = LookupTtable(dscore, pdiff_item1->nsupport+pdiff_item2->nsupport-2);
		}
		else
		{
			dpvalue = 1;
			dscore = 0;
		}
	}
	else //Mann-Whitney test
	{
		int nrank, ncount, n1, n2, n;
		double dsum1, dsum2, drank;
		double du, dse;

		if(gntgt_attr_type==ORDINAL)
		{
			dsum1 = 0;
			dsum2 = 0;
			nrank = 1;
			for(i=0;i<gnum_of_tgt_values;i++)
			{
				ncount = ((int*)pdiff_item1->ptgt_stat)[i]+((int*)pdiff_item2->ptgt_stat)[i];
				drank = (nrank+nrank+ncount-1)/2;
				dsum1 += drank*((int*)pdiff_item1->ptgt_stat)[i];
				dsum2 += drank*((int*)pdiff_item2->ptgt_stat)[i];
				nrank += ncount;
			}
		}
		else //CONTINUOUS 
		{
			n = 0;
			for(i=0;i<pdiff_item1->nsupport;i++)
			{
				gpgroup_tgt_values[n].ngroup_no = 0;
				gpgroup_tgt_values[n].dtgt_value = gptgt_values[pdiff_item1->ptid_list[i]];
				n++;
			}
			for(i=0;i<pdiff_item2->nsupport;i++)
			{
				gpgroup_tgt_values[n].ngroup_no = 1;
				gpgroup_tgt_values[n].dtgt_value = gptgt_values[pdiff_item2->ptid_list[i]];
				n++;
			}
			qsort(gpgroup_tgt_values, n, sizeof(GROUP_TGT_VALUE), comp_group_tgt_value);

			i = 0;
			nrank = 1;
			dsum1 = 0;
			dsum2 = 0;
			while(i<n)
			{
				ncount = 0;
				n1 = 0;
				n2 = 0;
				while(i+ncount<n && gpgroup_tgt_values[i+ncount].dtgt_value==gpgroup_tgt_values[i].dtgt_value)
				{
					ncount++;
					if(gpgroup_tgt_values[i+ncount].ngroup_no==0)
						n1++;
					else
						n2++;
				}
				drank = (nrank+nrank+ncount-1)/2;
				dsum1 += drank*n1;
				dsum2 += drank*n2;
				i += ncount;
				nrank += ncount;
			}
		}

		if(pdiff_item1->nsupport<pdiff_item2->nsupport || pdiff_item1->nsupport==pdiff_item2->nsupport && dsum1<=dsum2)
		{
			n1 = pdiff_item1->nsupport;
			n2 = pdiff_item2->nsupport;
		}
		else //if(pdiff_item1->nsupport>pdiff_item2->nsupport || pdiff_item1->nsupport==pdiff_item2->nsupport && dsum1>dsum2)
		{
			n1 = pdiff_item2->nsupport;
			n2 = pdiff_item1->nsupport;
			drank = dsum1;
			dsum1 = dsum2;
			dsum2 = drank;
		}
		if(n1+n2>MANN_WHITNEY_TEST_THRES)
		{
			du = (double)n1*(n1+n2+1)/2;
			dse = sqrt(n2*du/6);
			dscore = (dsum1-du)/dse;
			if(dscore<0)
				dscore = -dscore;
			dpvalue = LookupOneRow(dscore, gpdf1000_t_table, 20);
		}
		else
		{
			printf("Need to look up the table: %d %d %.3f\n", n1, n2, dsum1);
			dscore = 0;
			dpvalue = 0;
		}
	}

	return dpvalue;
}


//===============================================================================================
// calculate p-value for attribute level comparison
double CalcPvalue(PAT_INFO *ppat_info, DIFF_ITEM *pdiff_items, int num_of_diff_items, double &dscore, int &num_of_groups)
{
	int i, j, nsup_sum, nmin_group_size;
	double dtgt_sum, ddiff, dpvalue;

	nmin_group_size = gndb_size;
	nsup_sum = 0;
	for(i=0;i<num_of_diff_items;i++)
	{
		nsup_sum += pdiff_items[i].nsupport;
		if(nmin_group_size>pdiff_items[i].nsupport)
			nmin_group_size = pdiff_items[i].nsupport;
	}
	if(ppat_info->nsupport>nsup_sum)
		num_of_groups = num_of_diff_items+1;
	else
		num_of_groups = num_of_diff_items;

	if(num_of_groups==2)
	{
		dpvalue = CalcPvalue(&pdiff_items[0], &pdiff_items[1], dscore, false);
		return dpvalue;
	}

	if(gntgt_attr_type==NOMINAL || gntgt_attr_type==ORDINAL && (gsztarget_value[0]!=0 || gnum_of_tgt_values==2))
	{
		double dexp; 

		if(*((int*)ppat_info->ptgt_stat)==0 || *((int*)ppat_info->ptgt_stat)==ppat_info->nsupport)
		{
			dscore = 0;
			dpvalue = 1;
		}
		else if(gsztarget_value[0]!=0 || gnum_of_tgt_values==2)
		{
			dscore = 0;
			nsup_sum = 0;
			dtgt_sum = 0;
			for(i=0;i<num_of_diff_items;i++)
			{
				nsup_sum += pdiff_items[i].nsupport;
				dtgt_sum += ((int*)pdiff_items[i].ptgt_stat)[0];

				dexp = ((int*)ppat_info->ptgt_stat)[0]*(double)pdiff_items[i].nsupport/ppat_info->nsupport;
				ddiff = abs(((int*)pdiff_items[i].ptgt_stat)[0]-dexp)-0.5;
				if(dexp>0)
					dscore += ddiff*ddiff/dexp;

				dexp = (double)(ppat_info->nsupport-((int*)ppat_info->ptgt_stat)[0])*pdiff_items[i].nsupport/ppat_info->nsupport;
				ddiff = abs(pdiff_items[i].nsupport-((int*)pdiff_items[i].ptgt_stat)[0]-dexp)-0.5;
				if(dexp>0)
					dscore += ddiff*ddiff/dexp;
			}
			num_of_groups = num_of_diff_items;
			if(ppat_info->nsupport>nsup_sum)
			{
				dexp = ((int*)ppat_info->ptgt_stat)[0]*(ppat_info->nsupport-nsup_sum)/ppat_info->nsupport;
				ddiff  = abs(((int*)ppat_info->ptgt_stat)[0]-dtgt_sum-dexp)-0.5;
				if(dexp>0)
					dscore += ddiff*ddiff/dexp;

				dexp = (double)(ppat_info->nsupport-((int*)ppat_info->ptgt_stat)[0])*(ppat_info->nsupport-nsup_sum)/ppat_info->nsupport;
				ddiff = abs(ppat_info->nsupport-nsup_sum-(((int*)ppat_info->ptgt_stat)[0]-dtgt_sum)-dexp)-0.5;
				if(dexp>0)
					dscore += ddiff*ddiff/dexp;

				num_of_groups++;
			}

			dpvalue = LookupX2table(dscore, num_of_groups-1);
		}
		else
		{
			dscore = 0;
			nsup_sum = 0;
			for(i=0;i<num_of_diff_items;i++)
				nsup_sum += pdiff_items[i].nsupport;

			for(j=0;j<gnum_of_tgt_values;j++)
			{
				dtgt_sum = 0;
				for(i=0;i<num_of_diff_items;i++)
				{
					dtgt_sum += ((int*)pdiff_items[i].ptgt_stat)[j];

					dexp = ((int*)ppat_info->ptgt_stat)[j]*(double)pdiff_items[i].nsupport/ppat_info->nsupport;
					ddiff = abs(((int*)pdiff_items[i].ptgt_stat)[j]-dexp)-0.5;
					if(dexp>0)
						dscore += ddiff*ddiff/dexp;
				}
				if(ppat_info->nsupport>nsup_sum)
				{
					dexp = ((int*)ppat_info->ptgt_stat)[j]*(double)(ppat_info->nsupport-nsup_sum)/ppat_info->nsupport;
					ddiff = abs(((int*)ppat_info->ptgt_stat)[j]-dtgt_sum-dexp)-0.5;
					if(dexp>0)
						dscore += ddiff*ddiff/dexp; 
				}
			}
			num_of_groups = num_of_diff_items;
			if(ppat_info->nsupport>nsup_sum)
				num_of_groups++;
			dpvalue = LookupX2table(dscore, (num_of_groups-1)*(gnum_of_tgt_values-1));
		}
	}
	else if(gntgt_attr_type==CONTINUOUS_NORMAL || gntgt_attr_type==CONTINUOUS && gnmin_sup>=CLT_THRESHOLD) //nmin_group_size>=CLT_THRESHOLD)
	{
		// ANOVA
		double dbetween_group_score, dwithin_group_score, dmean, dmean_diff;

		nsup_sum = 0;
		dtgt_sum = 0;
		dbetween_group_score = 0;
		dwithin_group_score = ((TGT_SUM*)ppat_info->ptgt_stat)->dsquare_sum;
		for(i=0;i<num_of_diff_items;i++)
		{
			nsup_sum += pdiff_items[i].nsupport;
			dtgt_sum += ((TGT_SUM*)pdiff_items[i].ptgt_stat)->dsum;

			dmean_diff = pdiff_items[i].dmean-ppat_info->dmean;
			dbetween_group_score += pdiff_items[i].nsupport*dmean_diff*dmean_diff;
			dwithin_group_score -= ((TGT_SUM*)pdiff_items[i].ptgt_stat)->dsum*pdiff_items[i].dmean;
		}
		num_of_groups = num_of_diff_items;
		if(ppat_info->nsupport>nsup_sum)
		{
			num_of_groups++;

			dmean = (((TGT_SUM*)ppat_info->ptgt_stat)->dsum-dtgt_sum)/(ppat_info->nsupport-nsup_sum);
			dmean_diff = dmean-ppat_info->dmean;
			dbetween_group_score += (ppat_info->nsupport-nsup_sum)*dmean_diff*dmean_diff;
			dwithin_group_score -= (((TGT_SUM*)ppat_info->ptgt_stat)->dsum-dtgt_sum)*dmean;
		}
		dbetween_group_score /= (num_of_groups-1);
		dwithin_group_score /= (ppat_info->nsupport-num_of_groups);
		dscore = dbetween_group_score/dwithin_group_score;
		dpvalue = CalcANOVA(dscore, num_of_groups, ppat_info->nsupport-num_of_groups);

	}
	else if(num_of_diff_items==2)
	{
		num_of_groups = num_of_diff_items;

		dpvalue = CalcPvalue(&pdiff_items[0], &pdiff_items[1], dscore, false);
	}
	else //Kruskal-Wallis test
	{
		int nrank, ncount, N;
		double drank;

		num_of_groups = num_of_diff_items;

		memset(gpgroup_rank_avgs, 0, sizeof(double)*num_of_diff_items);
		if(gntgt_attr_type==ORDINAL)
		{
			nrank = 1;
			for(i=0;i<gnum_of_tgt_values;i++)
			{
				ncount = 0;
				for(j=0;j<num_of_diff_items;j++)
					ncount += ((int*)pdiff_items[j].ptgt_stat)[i];
				drank = (nrank+nrank+ncount-1)/2;
				for(j=0;j<num_of_diff_items;j++)
					gpgroup_rank_avgs[j] += drank*((int*)pdiff_items[j].ptgt_stat)[i];
				nrank += ncount;
			}
			N = 0;
			for(j=0;j<num_of_diff_items;j++)
				N += pdiff_items[j].nsupport;
			if(N+1!=nrank)
				printf("Error: inconsistent number of records\n");
		}
		else
		{
			N = 0;
			for(j=0;j<num_of_diff_items;j++)
			{
				for(i=0;i<pdiff_items[j].nsupport;i++)
				{
					gpgroup_tgt_values[N].ngroup_no = j;
					gpgroup_tgt_values[N].dtgt_value = gptgt_values[pdiff_items[j].ptid_list[i]];
					N++;
				}
			}
			qsort(gpgroup_tgt_values, N, sizeof(GROUP_TGT_VALUE), comp_group_tgt_value);

			i = 0;
			nrank = 1;
			while(i<N)
			{
				ncount = 0;
				while(i+ncount<N && gpgroup_tgt_values[i+ncount].dtgt_value==gpgroup_tgt_values[i].dtgt_value)
					ncount++;
				drank = (nrank+nrank+ncount-1)/2;
				j = 0;
				while(j<ncount)
				{
					gpgroup_rank_avgs[gpgroup_tgt_values[i+j].ngroup_no] += drank;
					j++;
				}
				i += ncount;
				nrank += ncount;
			}
		}

		dscore = 0;
		for(j=0;j<num_of_diff_items;j++)
			dscore += gpgroup_rank_avgs[j]/pdiff_items[j].nsupport*gpgroup_rank_avgs[j];
		dscore = dscore/N/(N+1)*12;
		dscore -= 3*(N+1);
		if(dscore<0)
			dscore = 0;
		dpvalue = LookupX2table(dscore, num_of_diff_items-1);
	}

	return dpvalue;
}

//=============  calculate p-value for association rules ================
double CalcRulePvalue(int nglobal_sup, char* pglobal_tgt_stat, int nXsup, char* pXtgt_stat, double &dscore, int *ptid_list, double *ptgt_values)
{
	double dmean_diff, dexpected, ddiff, dpvalue;
	int i;

	if(gntgt_attr_type==NOMINAL || gntgt_attr_type==ORDINAL && (gsztarget_value[0]!=0 || gnum_of_tgt_values==2))
	{
		int *ptgt_sups, *pglobal_tgt_sups;

		ptgt_sups = (int*)pXtgt_stat;
		pglobal_tgt_sups = (int*)pglobal_tgt_stat;

		if(gsztarget_value[0]!=0)
		{
			if(gntest_statisitic_method==X2_TEST)
			{		
				dexpected = (double)nXsup*pglobal_tgt_sups[0]/nglobal_sup;
				ddiff = abs(ptgt_sups[0]-dexpected)-0.5;
				if(dexpected>0)
					dscore = ddiff*ddiff/dexpected;

				dexpected = (double)nXsup*(nglobal_sup-pglobal_tgt_sups[0])/nglobal_sup;
				ddiff = abs(nXsup-ptgt_sups[0]-dexpected)-0.5;
				if(dexpected>0)
					dscore += ddiff*ddiff/dexpected;

				dexpected = (double)(nglobal_sup-nXsup)*pglobal_tgt_sups[0]/nglobal_sup;
				ddiff = abs(pglobal_tgt_sups[0]-ptgt_sups[0]-dexpected)-0.5;
				if(dexpected>0)
					dscore += ddiff*ddiff/dexpected;

				dexpected = (double)(nglobal_sup-nXsup)*(nglobal_sup-pglobal_tgt_sups[0])/nglobal_sup;
				ddiff = abs(nglobal_sup-nXsup-(pglobal_tgt_sups[0]-ptgt_sups[0])-dexpected)-0.5;
				if(dexpected>0)
					dscore += ddiff*ddiff/dexpected;

				dpvalue = LookupX2table(dscore, 1);
			}
			else
			{
				if(nglobal_sup<gndb_size)
					//dpvalue = CalcTwoTailedFisherPvalue(gndb_size, ((int*)gptgt_stat_array)[0], nXsup, ptgt_sups[0]);
					dpvalue = CalcFisherPvalue(nglobal_sup, pglobal_tgt_sups[0], nXsup, ptgt_sups[0]);
				else if(nXsup>=gnmin_sup)
					dpvalue = CalcTwoTailedFisherPvalue(nXsup, ptgt_sups[0]);
				else 
					dpvalue = CalcTwoTailedFisherPvalue(nglobal_sup, pglobal_tgt_sups[0], nXsup, ptgt_sups[0]);
					//dpvalue = CalcFisherPvalue(nXsup, ptgt_sups[0]);
				dscore = 0;
			}
		}
		else 
		{
			if(gnum_of_tgt_values==2 && gntest_statisitic_method==FISHER_EXACT_TEST)
			{
				dpvalue = CalcTwoTailedFisherPvalue(gndb_size, ((int*)gptgt_stat_array)[0], nXsup, ptgt_sups[0]);
				dscore = 0;
			}
			else
			{
				dscore = 0;
				for(i=0;i<gnum_of_tgt_values;i++)
				{				
					dexpected = (double)nXsup*pglobal_tgt_sups[i]/nglobal_sup;
					ddiff = abs(ptgt_sups[i]-dexpected)-0.5;
					if(dexpected>0)
						dscore += ddiff*ddiff/dexpected;

					dexpected = (double)(nglobal_sup-nXsup)*pglobal_tgt_sups[i]/nglobal_sup;
					ddiff = abs(pglobal_tgt_sups[i]-ptgt_sups[i]-dexpected)-0.5;
					if(dexpected>0)
						dscore += ddiff*ddiff/dexpected;
				}
				dpvalue = LookupX2table(dscore, gnum_of_tgt_values-1);
			}
		}
	}
	else if(gntgt_attr_type==CONTINUOUS_NORMAL || gntgt_attr_type==CONTINUOUS && nXsup>=CLT_THRESHOLD && nglobal_sup-nXsup>=CLT_THRESHOLD)
	{
		TGT_SUM *ptgt_sum, *pglobal_tgt_sum;
		double dvariance1, dvariance2, dmean, dtemp; 

		ptgt_sum = (TGT_SUM*)pXtgt_stat;
		pglobal_tgt_sum = (TGT_SUM*)pglobal_tgt_stat;

		dmean = (pglobal_tgt_sum->dsum-ptgt_sum->dsum)/(nglobal_sup-nXsup);
		dmean_diff = ptgt_sum->dsum/nXsup-dmean;
		if(dmean_diff<0)
			dmean_diff = -dmean_diff;
		dvariance1 = ptgt_sum->dsquare_sum-ptgt_sum->dsum*ptgt_sum->dsum/nXsup;
		dvariance2 = (pglobal_tgt_sum->dsquare_sum-ptgt_sum->dsquare_sum)-(pglobal_tgt_sum->dsum-ptgt_sum->dsum)*dmean;
		dtemp = (dvariance1+dvariance2)/(nglobal_sup-2);
		dtemp *= (1/(double)nXsup + 1/(double)(nglobal_sup-nXsup));
		dscore = dmean_diff/sqrt(dtemp);

		dpvalue = LookupTtable(dscore, nglobal_sup-2);
	}
	else //Mann-Whitney test
	{
		int nrank, ncount, n1, n2, n;
		double dsum1, dsum2, drank;
		double du, dse;

		if(gntgt_attr_type==ORDINAL)
		{
			dsum1 = 0;
			dsum2 = 0;
			nrank = 1;
			for(i=0;i<gnum_of_tgt_values;i++)
			{
				ncount = ((int*)pglobal_tgt_stat)[i];
				drank = (nrank+nrank+ncount-1)/2;
				dsum1 += drank*((int*)pXtgt_stat)[i];
				dsum2 += drank*(ncount-((int*)pXtgt_stat)[i]);
				nrank += ncount;
			}
		}
		else
		{
			n = 0;
			for(i=0;i<nglobal_sup;i++)
			{
				gpgroup_tgt_values[n].ngroup_no = 1;
				gpgroup_tgt_values[n].dtgt_value = ptgt_values[i];
				n++;
			}
			for(i=0;i<nXsup;i++)
				gpgroup_tgt_values[ptid_list[i]].ngroup_no = 0;
			qsort(gpgroup_tgt_values, n, sizeof(GROUP_TGT_VALUE), comp_group_tgt_value);

			i = 0;
			nrank = 1;
			dsum1 = 0;
			dsum2 = 0;
			while(i<n)
			{
				ncount = 0;
				n1 = 0;
				n2 = 0;
				while(i+ncount<n && gpgroup_tgt_values[i+ncount].dtgt_value==gpgroup_tgt_values[i].dtgt_value)
				{
					ncount++;
					if(gpgroup_tgt_values[i+ncount].ngroup_no==0)
						n1++;
					else
						n2++;
				}
				drank = (nrank+nrank+ncount-1)/2;
				dsum1 += drank*n1;
				dsum2 += drank*n2;
				i += ncount;
				nrank += ncount;
			}
		}

		if(nXsup<nglobal_sup-nXsup || nXsup==nglobal_sup-nXsup && dsum1<=dsum2)
		{
			n1 = nXsup;
			n2 = nglobal_sup-nXsup;
		}
		else 
		{
			n1 = nglobal_sup-nXsup;
			n2 = nXsup;
			drank = dsum1;
			dsum1 = dsum2;
			dsum2 = drank;
		}
		if(n1+n2>MANN_WHITNEY_TEST_THRES)
		{
			du = (double)n1*(n1+n2+1)/2;
			dse = sqrt(n2*du/6);
			dscore = (dsum1-du)/dse;
			if(dscore<0)
				dscore = -dscore;
			dpvalue = LookupOneRow(dscore, gpdf1000_t_table, 20);
		}
		else
		{
			printf("Need to look up the table: %d %d %.3f\n", n1, n2, dsum1);
			dscore = 0;
			dpvalue = 1;
		}
	}

	return dpvalue;
}

double CalcRulePvalue(int nglobal_sup, char* pglobal_tgt_stat, int nXsup, char* pXtgt_stat, double &dscore)
{
	double dmean_diff, dexpected, ddiff, dpvalue;
	int i;

	if(gntgt_attr_type==NOMINAL || gntgt_attr_type==ORDINAL && (gsztarget_value[0]!=0 || gnum_of_tgt_values==2))
	{
		int *ptgt_sups, *pglobal_tgt_sups;

		ptgt_sups = (int*)pXtgt_stat;
		pglobal_tgt_sups = (int*)pglobal_tgt_stat;

		if(gsztarget_value[0]!=0)
		{
			if(gntest_statisitic_method==X2_TEST)
			{		
				dexpected = (double)nXsup*pglobal_tgt_sups[0]/nglobal_sup;
				ddiff = abs(ptgt_sups[0]-dexpected)-0.5;
				if(dexpected>0)
					dscore = ddiff*ddiff/dexpected;

				dexpected = (double)nXsup*(nglobal_sup-pglobal_tgt_sups[0])/nglobal_sup;
				ddiff = abs(nXsup-ptgt_sups[0]-dexpected)-0.5;
				if(dexpected>0)
					dscore += ddiff*ddiff/dexpected;

				dexpected = (double)(nglobal_sup-nXsup)*pglobal_tgt_sups[0]/nglobal_sup;
				ddiff = abs(pglobal_tgt_sups[0]-ptgt_sups[0]-dexpected)-0.5;
				if(dexpected>0)
					dscore += ddiff*ddiff/dexpected;

				dexpected = (double)(nglobal_sup-nXsup)*(nglobal_sup-pglobal_tgt_sups[0])/nglobal_sup;
				ddiff = abs(nglobal_sup-nXsup-(pglobal_tgt_sups[0]-ptgt_sups[0])-dexpected)-0.5;
				if(dexpected>0)
					dscore += ddiff*ddiff/dexpected;

				dpvalue = LookupX2table(dscore, 1);
			}
			else
			{
				if(nglobal_sup<gndb_size)
					dpvalue = CalcTwoTailedFisherPvalue(nglobal_sup, pglobal_tgt_sups[0], nXsup, ptgt_sups[0]);
				else
					dpvalue = CalcTwoTailedFisherPvalue(nXsup, ptgt_sups[0]);
					//dpvalue = CalcFisherPvalue(nXsup, ptgt_sups[0]);
				dscore = 0;
			}
		}
		else 
		{
			if(gnum_of_tgt_values==2 && gntest_statisitic_method==FISHER_EXACT_TEST)
			{
				dpvalue = CalcTwoTailedFisherPvalue(gndb_size, ((int*)gptgt_stat_array)[0], nXsup, ptgt_sups[0]);
				//dpvalue = CalcTwoTailedFisherPvalue(nXsup, ptgt_sups[0]);
				dscore = 0;
			}
			else
			{
				dscore = 0;
				for(i=0;i<gnum_of_tgt_values;i++)
				{				
					dexpected = (double)nXsup*pglobal_tgt_sups[i]/nglobal_sup;
					ddiff = abs(ptgt_sups[i]-dexpected)-0.5;
					if(dexpected>0)
						dscore += ddiff*ddiff/dexpected;

					dexpected = (double)(nglobal_sup-nXsup)*pglobal_tgt_sups[i]/nglobal_sup;
					ddiff = abs(pglobal_tgt_sups[i]-ptgt_sups[i]-dexpected)-0.5;
					if(dexpected>0)
						dscore += ddiff*ddiff/dexpected;
				}
				dpvalue = LookupX2table(dscore, gnum_of_tgt_values-1);
			}
		}
	}
	else if(gntgt_attr_type==CONTINUOUS_NORMAL || gntgt_attr_type==CONTINUOUS && nXsup>=CLT_THRESHOLD && nglobal_sup-nXsup>=CLT_THRESHOLD)
	{
		TGT_SUM *ptgt_sum, *pglobal_tgt_sum;
		double dvariance1, dvariance2, dmean, dtemp; 

		ptgt_sum = (TGT_SUM*)pXtgt_stat;
		pglobal_tgt_sum = (TGT_SUM*)pglobal_tgt_stat;

		dmean = (pglobal_tgt_sum->dsum-ptgt_sum->dsum)/(nglobal_sup-nXsup);
		dmean_diff = ptgt_sum->dsum/nXsup-dmean;
		if(dmean_diff<0)
			dmean_diff = -dmean_diff;
		dvariance1 = ptgt_sum->dsquare_sum-ptgt_sum->dsum*ptgt_sum->dsum/nXsup;
		dvariance2 = (pglobal_tgt_sum->dsquare_sum-ptgt_sum->dsquare_sum)-(pglobal_tgt_sum->dsum-ptgt_sum->dsum)*dmean;
		dtemp = (dvariance1+dvariance2)/(nglobal_sup-2);
		dtemp *= (1/(double)nXsup + 1/(double)(nglobal_sup-nXsup));
		dscore = dmean_diff/sqrt(dtemp);

		dpvalue = LookupTtable(dscore, nglobal_sup-2);
	}
	else //Mann-Whitney test
	{
		int nrank, ncount, n1, n2;
		double dsum1, dsum2, drank;
		double du, dse;

		if(gntgt_attr_type==ORDINAL)
		{
			dsum1 = 0;
			dsum2 = 0;
			nrank = 1;
			for(i=0;i<gnum_of_tgt_values;i++)
			{
				ncount = ((int*)pglobal_tgt_stat)[i];
				drank = (nrank+nrank+ncount-1)/2;
				dsum1 += drank*((int*)pXtgt_stat)[i];
				dsum2 += drank*(ncount-((int*)pXtgt_stat)[i]);
				nrank += ncount;
			}

			if(nXsup<nglobal_sup-nXsup || nXsup==nglobal_sup-nXsup && dsum1<=dsum2)
			{
				n1 = nXsup;
				n2 = nglobal_sup-nXsup;
			}
			else 
			{
				n1 = nglobal_sup-nXsup;
				n2 = nXsup;
				drank = dsum1;
				dsum1 = dsum2;
				dsum2 = drank;
			}
			if(n1+n2>MANN_WHITNEY_TEST_THRES)
			{
				du = (double)n1*(n1+n2+1)/2;
				dse = sqrt(n2*du/6);
				dscore = (dsum1-du)/dse;
				if(dscore<0)
					dscore = -dscore;
				dpvalue = LookupOneRow(dscore, gpdf1000_t_table, 20);
			}
			else
			{
				printf("Need to look up the table: %d %d %.3f\n", n1, n2, dsum1);
				dscore = 0;
				dpvalue = 0;
			}

		}
		else
		{
			dscore = 0;
			dpvalue = 1;
			//printf("Not implemented yet\n");
		}
	}

	return dpvalue;
}


