//-----------------------------------------------
// Copyright 2016 Guangxi University
// Written by Liang Zhao(S080011@e.ntu.edu.sg)
// Released under the GPL
//-----------------------------------------------
//
// index - Build a BWT/FM-index for a set of reads
// It's developed based on SGA, originally writen by Jared Simpson (js18@sanger.ac.uk)
//

#include <iostream>
#include <fstream>
#include <algorithm>
#include "MECCommon.h"
#include "Util.h"
#include "index.h"
#include "SuffixArray.h"
#include "SeqReader.h"
#include "SACAInducedCopying.h"
#include "BWT.h"
#include "Timer.h"
#include "BWTCARopebwt.h"
#include "SampledSuffixArray.h"

//
// Getopt
//
#define SUBPROGRAM "index"

static const char *INDEX_VERSION_MESSAGE =
SUBPROGRAM " Version " PACKAGE_VERSION "\n";

static const char *INDEX_USAGE_MESSAGE =
"Usage: " PACKAGE_NAME " " SUBPROGRAM " [OPTION] ... READSFILE\n"
"\n"
"        --help             display this help and exit\n"
"    -t, --threads=NUM      use NUM threads to construct the index (default: 1)\n"
"    -p, --prefix=PREFIX    write index to file using PREFIX instead of prefix of READSFILE\n"
"        --no-reverse       suppress construction of the reverse BWT. Use this option when building the index\n"
"                           for reads that will be error corrected using the k-mer corrector, which only needs the forward index\n"
"        --no-forward       suppress construction of the forward BWT. Use this option when building the forward and reverse index separately\n"
"    -g, --gap-array=N      use N bits of storage for each element of the gap array. Acceptable values are 4,8,16 or 32. Lower\n"
"                           values can substantially reduce the amount of memory required at the cost of less predictable memory usage.\n"
"                           When this value is set to 32, the memory requirement is essentially deterministic and requires ~5N bytes where\n"
"                           N is the size of the FM-index of READS2.\n"
"                           The default value is 8.\n"
"\nReport bugs to " PACKAGE_BUGREPORT "\n\n";

namespace opt
{
    static std::string readsFile;
    static std::string prefix;
    static std::string algorithm = "ropebwt";
    static int numReadsPerBatch = 2000000;
    static int numThreads = 1;
    static bool bDiskAlgo = false;
    static bool bBuildReverse = true;
    static bool bBuildForward = true;
    static bool bBuildSAI = true;
    static int gapArrayStorage = 4;
}

static const char* shortopts = "p:t:g";

enum { OPT_HELP = 1, OPT_VERSION, OPT_NO_REVERSE, OPT_NO_FWD, OPT_NO_SAI };

static const struct option longopts[] = {
    { "prefix",      required_argument, NULL, 'p' },
    { "threads",     required_argument, NULL, 't' },
    { "gap-array",   required_argument, NULL, 'g' },
    { "no-reverse",  no_argument,       NULL, OPT_NO_REVERSE },
    { "no-forward",  no_argument,       NULL, OPT_NO_FWD },
    { "no-sai",      no_argument,       NULL, OPT_NO_SAI },
    { "help",        no_argument,       NULL, OPT_HELP },
    { "version",     no_argument,       NULL, OPT_VERSION },
    { NULL, 0, NULL, 0 }
};

int indexMain(int argc, char** argv)
{
    Timer t("index");
    parseIndexOptions(argc, argv);
    indexInMemoryRopebwt();
    return 0;
}

//
void indexInMemoryRopebwt()
{
    std::cout << "Building index for " << opt::readsFile << " in memory using ropebwt\n";

    bool use_threads = opt::numThreads >= 4;

    if(opt::bBuildForward)
    {
        std::string bwt_filename = opt::prefix + BWT_EXT;
        std::string sai_filename = opt::prefix + SAI_EXT;
        BWTCA::runRopebwt(opt::readsFile, bwt_filename, use_threads, false);

        if(opt::bBuildSAI)
        {
            std::cout << "\t done bwt construction, generating .sai file\n";
            BWT* pBWT = new BWT(bwt_filename);
            SampledSuffixArray ssa;
            ssa.buildLexicoIndex(pBWT, opt::numThreads);
            ssa.writeLexicoIndex(sai_filename);
            delete pBWT;
        }
    }

    if(opt::bBuildReverse)
    {
        std::string rbwt_filename = opt::prefix + RBWT_EXT;
        std::string rsai_filename = opt::prefix + RSAI_EXT;
        BWTCA::runRopebwt(opt::readsFile, rbwt_filename, use_threads, true);

        if(opt::bBuildSAI)
        {
            std::cout << "\t done rbwt construction, generating .rsai file\n";
            BWT* pRBWT = new BWT(rbwt_filename);
            SampledSuffixArray ssa;
            ssa.buildLexicoIndex(pRBWT, opt::numThreads);
            ssa.writeLexicoIndex(rsai_filename);
            delete pRBWT;
        }
    }
}

// 
// Handle command line arguments
//
void parseIndexOptions(int argc, char** argv)
{
    bool die = false;
    for (char c; (c = getopt_long(argc, argv, shortopts, longopts, NULL)) != -1;) 
    {
        std::istringstream arg(optarg != NULL ? optarg : "");
        switch (c) 
        {
            case 'p': arg >> opt::prefix; break;
            case '?': die = true; break;
            case 't': arg >> opt::numThreads; break;
            case 'g': arg >> opt::gapArrayStorage; break;
            case OPT_NO_REVERSE: opt::bBuildReverse = false; break;
            case OPT_NO_FWD: opt::bBuildForward = false; break;
            case OPT_NO_SAI: opt::bBuildSAI = false; break;
            case OPT_HELP:
                std::cout << INDEX_USAGE_MESSAGE;
                exit(EXIT_SUCCESS);
            case OPT_VERSION:
                std::cout << INDEX_VERSION_MESSAGE;
                exit(EXIT_SUCCESS);
        }
    }

    // Transform algorithm parameter to lower case
    std::transform(opt::algorithm.begin(), opt::algorithm.end(), opt::algorithm.begin(), ::tolower);
    
    if (argc - optind < 1) 
    {
        std::cerr << SUBPROGRAM ": missing arguments\n";
        die = true;
    } 
    else if (argc - optind > 1) 
    {
        std::cerr << SUBPROGRAM ": too many arguments\n";
        die = true;
    }

    if(opt::gapArrayStorage != 4 && opt::gapArrayStorage != 8 &&
       opt::gapArrayStorage != 16 && opt::gapArrayStorage != 32)
    {
        std::cerr << SUBPROGRAM ": invalid argument, --gap-array,-g must be one of 4,8,16,32 (found: " << opt::gapArrayStorage << ")\n";
        die = true;
    }

    if(opt::numThreads <= 0)
    {
        std::cerr << SUBPROGRAM ": invalid number of threads: " << opt::numThreads << "\n";
        die = true;
    }

    if (die) 
    {
        std::cout << "\n" << INDEX_USAGE_MESSAGE;
        exit(EXIT_FAILURE);
    }

    // Parse the input filenames
    opt::readsFile = argv[optind++];
    if(opt::prefix.empty())
        opt::prefix = stripFilename(opt::readsFile);

    // Check if input file is empty
    size_t filesize = getFilesize(opt::readsFile);
    if(filesize == 0)
    {
        std::cerr << SUBPROGRAM ": input file is empty\n";
        exit(EXIT_FAILURE);
    }
}
